% SETAXESNYQUIST Set x- and y- axis limits such that all plotted data is 
%  contained in the plot frame and the axes is square (1:1).
%  QUADPROG is employed to select the limits optimally.
%
% Use this function with THESISFORMAT. Otherwise, you need to specify 
% the aspect ratio of the plot frame manually.
%
% Call after plotting, but before THESISFORMAT. Example:
% 
%   figure; 
%   plot(xdata,ydata);
%   setAxisNyquist;
%   thesisFormat;
%
% Detailed usage description:
%
% limits = SETAXESNYQUIST() sets the x- and y- axis limits such that
%   all of the plotted data is contained and the scale is square (1:1) such
%   that the plotted geometry (e.g. curvature) is true. LIMITS is a
%   4-vector of the limits in the format [xmin;xmax;ymin;ymax].
%
% limits = SETAXESNYQUIST(...,'aspect',a) uses aspect ratio A instead of
%   inferring the value from the axes UserData (if the axes is part of
%   a thesisFigure, otherwise using the golden ratio).
%
% limits = SETAXESNYQUIST(...,'edges',method) controls how the plot limits
%   are computed. The choices are:
%      'inframe': All plotted data is required to remain in the plot frame.
%                 quadprog will be called to compute the limits. (DEFAULT)
%      'close'  : Most, but possibly not all, data will be the plot frame.
%                 A closed-form solution is used that does not require fmincon.
%
% limits = SETAXESNYQUIST(...,'padxPct',px) explicitly sets the x-padding in
%   percent.
%
% limits = SETAXESNYQUIST(...,'padyPct',py) explicitly sets the y-padding in
%   percent.
% 
% limits = SETAXESNYQUIST(...,'axes',ax) uses axes handle AX instead of gca.
%
% limits = SETAXESNYQUIST(...,'xdata',xd) uses the x-values in the vector or 
%   matrix XD to determine the x-span of the plotted data instead of
%   extracting the data-values from the supplied axis handle.
%
% limits = SETAXESNYQUIST(...,'ydata',yd) uses the y-values in the vector or 
%   matrix YD to determine the y-span of the plotted data instead of
%   extracting the data-values from the supplied axis handle.
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function x = setAxesNyquist(varargin)
  parser = inputParser;
  parser.addParameter('aspect',[],@(x)isscalar(x)&&x>0);
  parser.addParameter( ...
      'edges','inframe',@(x)strcmp(x,'close')||strcmp(x,'inframe'));
  parser.addParameter('padxPct',3,@(x)isscalar(x)&&x>=0);
  parser.addParameter('padyPct',3,@(x)isscalar(x)&&x>=0);
  parser.addParameter('axes',[]);
  parser.addParameter('xdata',[]);
  parser.addParameter('ydata',[]);
  parser.parse(varargin{:});
  p = parser.Results;  % structure of validated arguments.
  edges = p.edges;
  padxPct = p.padxPct;
  padyPct = p.padyPct;
  ax = p.axes;
  if isempty(ax)
    ax = gca();
  end
  fig = ax.Parent;
  xdata = p.xdata;
  ydata = p.ydata;
  
  % Determine aspect ratio.
  if isempty(p.aspect)
    if isstruct(fig.UserData) && ...
      isfield(fig.UserData,'origin__') && ...
      strcmp(fig.UserData.origin__,'thesisfig')
      % Figure generated by thesisFigure; use appropriate aspect ratio.
      aspect = fig.UserData.param.AxesAspectRatio;
    else
      aspect = 2/(sqrt(5)-1);  % golden ratio (width/height)
    end
  else
    aspect = parser.Results.aspect;
  end
  
  % Collect x/y data points.
  lines = findall(ax,'Type','line');
  if isempty(xdata)
    xdata = [lines.XData];
  end
  if isempty(ydata)
    ydata = [lines.YData];
  end
  
  % Remove nans!
  xdata = xdata(~isnan(xdata));
  ydata = ydata(~isnan(ydata));
  
  % Check for empty vectors.
  if isempty(xdata) || isempty(ydata)
    warning('setAxesNyquist: Couldn''t determine x,y plot limits!');
    x = [Nan Nan NaN NaN];
    return;
  end
  
  xmin = min(xdata,[],'all'); xmax = max(xdata,[],'all');
  ymin = min(ydata,[],'all'); ymax = max(ydata,[],'all');
  padx = (xmax-xmin)*padxPct/100;
  pady = (ymax-ymin)*padyPct/100;
  
  xL = [xmin-padx; xmax+padx; ymin-pady; ymax+pady];
  gv = [-1 1 aspect -aspect];
  
  % Lagrange solution to optimization problem:
  % Minimize J = (x-xL)'*(x-xL) over x subject to the
  % aspect ratio constraint gv*x = 0 where
  % x = [x1 x2 y1 y2]' represents the plot limits.
  x = xL-gv*xL*gv'/2/(1+aspect^2);
  
  if strcmp(edges,'inframe')
    % Explicit constrained optimization.

    H = 2*eye(length(xL));
    f = -2*xL;
    
    % Linear inequality: Ax <= B.
    % We require: x1 <= xmin, x2 >= xmax, y1 <= ymin, y2 >= ymax
    % where [x1 x2 y1 y2] are the plot limits.
    A = diag([1 -1 1 -1]);
    B = A*xL;

    % Linear equality: Aeq x = Beq.
    % We require: (x2 - x1) = aspect*(y2 - y1)
    % where [x1 x2 y1 y2] are the plot limits.
    Aeq = gv;
    Beq = 0;

    % Create initial guess.
    % Will be biased to pad toward the top.
    x0_1 = min(xmin,x(1));
    x0_2 = max(xmax,x(2));
    y0_1 = min(ymin,x(3));
    y0_2 = y0_1+(x0_2-x0_1)/aspect;
    x0 = [x0_1; x0_2; y0_1; y0_2];

    % Search for solution.
    options = optimoptions('quadprog','Display','off');
    x = quadprog(H,f,A,B,Aeq,Beq,[],[],x0,options);
  end
  
  % Apply computed limits to the axis.
  axis(ax,x);
end