% PROCESSPULSE Preprocess raw pulse data, fitting an equivalent-circuit
% model (ECM) to the pulse response at each temperature, SOC, C-rate,
% and repeat index.
%
% -- Usage --
% cellData = processPulse(cellspec,config)
%
% -- Input --
% cellspec     = cell specification generated by the labcell() function
% config       = structure with the following configuration parameters:
%   .ss        = 2-tuple of time bounds for steady-state interval [s]
%   .relax     = 2-tuple of time bounds for relaxation interval [s]
%   .regress   = 2-tuple of time bounds for regression interval [s]
%   .vtpct     = threshold voltage as percent of peak over/undershoot [%]
%   .getWeight = handle to function that returns residual weight vector
%                given the time vector as an argument
%   .postProcessECM = Handle to function that post-processes ECM data at
%                a given temperature (OPTIONAL). Accepts a structure 
%                with the following fields, and returns the same structure,
%                possibly modified:
%     .{ecmParamName}   = matrix of values of a particular ECM
%                         (dim1=SOC, dim2=iapp)
%     .sig{ecmParamName}= matrix of standard deviations of a particular
%                         ECM parameter over repeats
%     .socPct           = vector of actual SOC setpoints [%]
%     .I                = matrix of actual applied current
%                         (dim1=SOC, dim2=magnitude point)
%     .sigI             = matrix of standard deviations of I 
%                         over repeats
%     .iapp             = vector of requested iapp setpoints (not actual
%                         observed!)
%
% NOTE: time=0 corresponds to the onset of the pulse current
%
% -- Output --
% cellData = struct with the following fields:
%
% -- Output Files --
% [folder]/[cellname]_CELL/mat/
%   processPulse.mat = MAT file containing cellData structure
%
% -- Note --
% 1. This function runs considerably faster with plots disabled, i.e. use
%    `outp.normal` or `outp.quiet`.
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function processData = processPulse(cellspec,config,varargin)
  
  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addRequired('config',@(x)isstruct(x)&&isscalar(x));
  parser.parse(cellspec,config,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  cellname = arg.cellspec.name;
  lockfile = arg.cellspec.pls.lockfile;
  fitFileDischg = arg.cellspec.dis.fitfile;
  timestamp = arg.cellspec.timestamp__;
  
  outp.print('Started processPulse %s\n',cellname);
  
  % Load data from lockfile.
  lock = load(lockfile);
  matData = lock.matData;
  
  % Load cellData from dischg file.
  dataDischg = load(fitFileDischg);
  cellData = dataDischg.cellData;
  QAh = cellData.const.Q;  % total capacity for true SOC computation
  
  % Process pulse data:
  % * Compute true SOC at each setpoint
  % * Fit ECM to each pulse, average over repeats
  clear pulseVect;  % flat struct vector of pulses
  cursor = 1;       % index into the vector above
  % -- Iterate temperatures --
  for kt = 1:length(matData)
    dataT = matData(kt);
    socSeries = dataT.socSeries;
    socPct = dataT.soc0Pct; % initial SOC in percent
    % -- Iterate SOC setpoints --
    for kz = 1:length(socSeries)
      dataZ = socSeries(kz);
      if isfield(dataZ.dischg,'iapp')
        QdisAh = trapz(dataZ.dischg.time,dataZ.dischg.iapp)/3600;
        socPct = socPct - 100*QdisAh/QAh; % calculate true SOC in percent
      else
        % Missing discharge data - revert to setpoint.
        socPct = dataZ.socPct;
        outp.warning(['WARNING: Missing discharge data for ' ...
            '%.2fdegC / %.2fSOC. Using SOC setpoint directly instead ' ...
            'of coloumb counting.\n'],dataZ.TdegC,dataZ.socPct);
      end
      magnitudeSeries = dataZ.magnitudeSeries;
      % -- Iterate pulse magnitudes --
      for ki = 1:length(magnitudeSeries)
        dataI = magnitudeSeries(ki);
        repeatSeries = dataI.repeatSeries;
        % -- Iterate repeats --
        clear ecmParamValues;
        for kr = length(repeatSeries):-1:1
          pulseData = repeatSeries(kr);
          ecmParamValues(kr) = fitECM(pulseData,arg);
        end % for repeat
        % -- Average ECM parameter values over pulses --
        val = struct;  % struct of mean value for each param
        lb = struct;   % lower confidence interval (CI) bound
        ub = struct;   % upper CI bound
        paramNames = fieldnames(ecmParamValues);
        for kp = 1:length(paramNames)
          pname = paramNames{kp};
          values = [ecmParamValues.(pname)];
          if any(isinf(values))
            % Capacitance may be infinite if there is zero voltage
            % delta!
            val.(pname) = Inf;
            lb.(pname) = Inf;
            ub.(pname) = Inf;
          else
            PD = fitdist(values(:),'Normal'); % fit Gaussian dist.
            CI = paramci(PD,'Alpha',0.01); % 99% CI
            val.(pname) = mean(values);
            lb.(pname) = CI(1);
            ub.(pname) = CI(2);
          end
        end % for
        % -- Store results --
        tmp = struct;
        tmp.ecmval = val;
        tmp.ecmlb = lb;
        tmp.ecmub = ub;
        tmp.TdegC = dataI.TdegC;
        tmp.socPct = dataI.socPct;
        tmp.socTruePct = socPct;
        tmp.iapp = dataI.I;
        tmp.iappTrue = val.I;
        pulseVect(cursor) = tmp; %#ok<AGROW>
        cursor = cursor + 1;
        % -- Output to command window --
        outp.print('%5.2fdegC %5.2f%% %+6.2fA: ', ...
            tmp.TdegC,tmp.socTruePct,tmp.iappTrue);
        for kp = 1:length(paramNames)
          pname = paramNames{kp};
          if strcmp(pname,'I')
            continue; % already printed above!
          end
          mu = tmp.ecmval.(pname);
          outp.print('%3s = %+6.2e ',pname,mu);
        end % for
        outp.ln;
      end % for ki
    end % for kz
  end % for kt
  
  % Postprocess pulse data:
  % * Collect ECM parameters into matricies at each temperature
  ecmParamNames = fieldnames(pulseVect(1).ecmval);
  % -- Iterate temperatures --
  clear ecmVect;
  Tvect = unique([pulseVect.TdegC]);
  for kt = 1:length(Tvect)
    % Select all pulses at current temperature.
    TdegC = Tvect(kt);
    indT = TdegC==[pulseVect.TdegC];
    pulseVectT = pulseVect(indT);
    % Collect current magnitudes and SOC setpoints.
    [socPct,indz] = unique([pulseVectT.socPct]);
    socTruePct = [pulseVectT(indz).socTruePct];
    iapp = unique([pulseVectT.iapp]);
    nz = length(socPct);
    ni = length(iapp);
    % -- Iterate ECM parameters --
    tmp = struct;
    for kp = 1:length(ecmParamNames)
      pname = ecmParamNames{kp};
      val = zeros(nz,ni);
      lb = zeros(nz,ni);
      ub = zeros(nz,ni);
      for kz = 1:nz
        for ki = 1:ni
          % Find pulse at this SOC and current magnitude.
          ind = [pulseVectT.socPct]==socPct(kz) & ...
                [pulseVectT.iapp]==iapp(ki);
          pulse = pulseVectT(ind);
          val(kz,ki) = pulse.ecmval.(pname);
          lb(kz,ki) = pulse.ecmlb.(pname);
          ub(kz,ki) = pulse.ecmub.(pname);
        end % for ki
      end % for kz
      tmp.(pname) = val;
      tmp.(['lb' pname]) = lb;
      tmp.(['ub' pname]) = ub;
    end % for ECM param
    % (note: at this point, tmp.I contains the current magnitudes!)
    tmp.TdegC = TdegC;
    tmp.socPct = socTruePct;  % !! use true SOC vector
    tmp.iapp = iapp;
    if isfield(arg.config,'postProcessECM')
      pp = arg.config.postProcessECM(tmp);
    else
      pp = tmp;
    end
    ecmVect(kt) = pp; %#ok<AGROW>
    % -- Plot pulse resistance --
    if outp.plot
      lab = arrayfun(@(z)sprintf('%.2f%%',z),socTruePct, ...
          'UniformOutput',false);
      figure;
      subplot(121);
      cRate = repmat(tmp.iapp/cellData.const.Q,size(tmp.R,1),1);
      errorbar(cRate.',tmp.R.',(tmp.ubR-tmp.lbR).'/2, ...
          '.','MarkerSize',25,'CapSize',18);
      ylim(mean(tmp.R,'all')+[-1 1]*2*mean(tmp.ubR(tmp.R>0)-tmp.lbR(tmp.R>0),'all'));
      xlabel('Pulse Magnitude [C rate]');
      ylabel('Pulse Resistance, R_0 [\Omega]');
      title(sprintf('R_0 (T=%.2f\\circC initial)',TdegC));
      legend(lab,'Location','best');
      subplot(122);
      cRate = repmat(pp.iapp/cellData.const.Q,size(pp.R,1),1);
      errorbar(cRate.',pp.R.',(pp.ubR-pp.lbR).'/2, ...
          '.','MarkerSize',25,'CapSize',18);
      ylim(mean(tmp.R,'all')+[-1 1]*2*mean(tmp.ubR(tmp.R>0)-tmp.lbR(tmp.R>0),'all'));
      xlabel('Pulse Magnitude [C rate]');
      ylabel('Pulse Resistance, R_0 [\Omega]');
      title(sprintf('R_0 (T=%.2f\\circC post-processed)',TdegC))
      thesisFormat;
    end % if
  end % for kt
  
  % Collect and save output data.
  processData.cellData = cellData;
  processData.pulseVect = pulseVect;
  processData.ecmVect = ecmVect;
  processData.origin__ = 'processPulse';
  processData.arg__ = arg;
  processData.timestamp__ = timestamp;
  save(arg.cellspec.pls.processfile,'-struct','processData');
  
  outp.print('Finished processPulse %s\n\n',cellname);
end


% Utility functions -------------------------------------------------------

% FITECM Fit equivalent-circuit model to pulse waveform.
function out = fitECM(pulseData,arg)

  % Number of units, i.e. independent Gamry devices connected in parallel.
  Nu = size(pulseData.pulse.iapp,2);
  
  % Initialize plot showing the result of each ECM regression.
  persistent fig lines_vcellLAB lines_vcellMOD plotTitle;
  if outp.debug && (isempty(fig) || ~ishandle(fig))
    fig = figure( ...
        'Name','ECM Regression', ...
        'WindowStyle','docked');
    for k = 1:2
      col = cool(Nu);
      lab1 = arrayfun(@(x)sprintf('Lab (u%d)',x),1:Nu,'UniformOutput',false);
      lab2 = arrayfun(@(x)sprintf('Model (u%d)',x),1:Nu,'UniformOutput',false);
      for ku = 1:Nu
        lines_vcellLAB(ku) = plot(NaN,NaN,':','Color',col(ku,:)); hold on;
      end
      for ku = 1:Nu
        lines_vcellMOD(ku) = plot(NaN,NaN,'Color',col(ku,:)); hold on;
      end
      xlabel('Time, $t$ [$\mu\mathrm{s}$]', ...
          'Interpreter','latex');
      ylabel('$v_\mathrm{cell}-U_\mathrm{ocv}$ [$\mathrm{V}$]', ...
          'Interpreter','latex');
      plotTitle = title('ECM Regression','FontName','Monospace');
      legend(lab1{:},lab2{:},'Location','north','NumColumns',2);
      thesisFormat;
      drawnow;
    end % for
  end % if
  
  % Collect data.
  config = arg.config;
  pulse = pulseData.pulse;   % structure with time, iapp, vcell
  time = pulse.time(:,1);    % time vector from master unit
  iapp = pulse.iapp;         % matrix of iapp column vectors for each unit
  vcell = pulse.vcell;       % matrix of vcell column vectors for each unit
  ocv = mean(pulseData.ocv.vcell); % open-circuit voltage at this setpoint
  ind0 = find(time==0);      % sample index at which time is zero
  ss = config.ss(1)<=time&time<=config.ss(2); % logical indicies to ss interval
  relax = config.relax(1)<=time&time<=config.relax(2); % logical indicies to relaxation interval
  regress = config.regress(1)<=time&time<=config.regress(2); % logical indicies to regression interval
  NactiveUnits = sum(all(~isnan(vcell)));
  
  % -- Synchronize the voltage waveforms -- 
  % * Ensure the rising edges occur at the same sample instant
  if NactiveUnits>1
    vss = mean(vcell(ss,1));     % primary unit's vcell in ss interval
    vt = (vss-ocv)*config.vtpct/100;   % edge threshold voltage at which to synchronize
    for k = 1:size(vcell,2)
      vunit = vcell(:,k);
      if all(isnan(vunit))
        % This unit disabled for this pulse.
        continue;
      end % if
      indEdge = find((vunit-ocv)*sign(vt)>abs(vt),1,'first');
      % Align edge position to the sample index just after t=0.
      offset = (ind0 + 1) - indEdge;
      if 1<=offset && offset<=2
        vcell(:,k) = [vunit(1:offset); vunit(1:end-offset)];
      end % if
    end % for
  end
  
  % -- Determine series capacitance for each unit using Q=CV --
  ocvi = median(vcell(time<0,:)); % initial voltages (row vector!)
  ocvf = median(vcell(relax,:));  % final voltages (row vector!)
  q = cumtrapz(time,-iapp);       % charge added(+)/removed(-) to/from cap
  C = q(end,:)./(ocvf - ocvi); % equivalent series capacitances (row vector!)
  % Subtract out initial voltage and voltage change due to series cap.
  vnorm = vcell - ocvi - q./C;
  
  % -- Window the waveforms for the regression --
  t = time(regress);     % time   [s]
  i = -iapp(regress,:);  % -iapp  [A]  !!! invert sign for ECM regression
  q = cumtrapz(t,i);     % Q      [C]
  v = vnorm(regress,:);  % vcell [ V]
  cv = cumtrapz(t,v);    % integral(vcell) [V s]
  W = diag(config.getWeight(t));  % residual weighting matrix
  
  % -- Least-squares linear regression to fit ECM to pulse --
  % * Series R-L circuit, one "R" and one "L" per branch / use all 3 voltages
  Nb = size(v,2);
  R = zeros(Nb,1);
  L = zeros(Nb,1);
  for k = 1:Nb
    if all(isnan(v(:,k)))
      % Unit disabled.
      R(k) = NaN;
      L(k) = NaN;
      continue;
    end
    % Define linear system of the form y = H*x
    H = [q(:,k) i(:,k)]; % measurement matrix
    Y = cv(:,k);         % observation vector
    % Compute weighted LS solution
    x = (sqrt(W)*H)\(sqrt(W)*Y); % vector of unknowns (R and L)
    R(k) = x(1);
    L(k) = x(2);
  end % for
  
  % Compute model prediction.
  vmodel = i*diag(R) + [zeros(1,Nb); diff(i)./diff(t)]*diag(L);
  
  % Store output.
  out.R = mean(R,'omitnan')/NactiveUnits;
  out.L = mean(L,'omitnan')/NactiveUnits;
  out.C = mean(C,'omitnan')/NactiveUnits;
  out.I = mean(sum(iapp(ss,:),2)); % estimate of total current in ss interval
  
  % Update plot.
  if outp.debug
    for ku = 1:Nu
      set(lines_vcellLAB(ku),'XData',t*1e6,'Ydata',v(:,ku));
      set(lines_vcellMOD(ku),'XData',t*1e6,'Ydata',vmodel(:,ku));
      set(plotTitle,'String', ...
          sprintf('%2.0fdegC %2d%% %4.0fA #%2d', ...
              pulseData.TdegC,pulseData.socPct,pulseData.I,pulseData.repeat));
    end
    drawnow;
  end

end