%PROCESSOCV Preprocess raw OCV data from full cell. 
%
% * Compute total capacity (Q) and couloumbic efficiency (eta) @ each temp
% * Convert OCV-vs-time data to OCV-vs-SOC data @ each temp
% * Smooth OCV-vs-SOC curves and compute differential capacity @ each temp
%
% -- Usage --
% cellData = processOCV(cellspec)
% cellData = processOCV(...,'dv',dv)
%
% -- Input --
% cellspec = cell specification generated by labcell() function
% dv       = voltage resolution for voltage-v-soc and differential 
%            capacity computation [V]
%
% -- Output --
% cellData = struct with the following fields:
% 
%   raw : raw data @ each temp (struct array)
%   cal : calibrated data @ each temp  (struct array)
%
% -- Output Files --
% [folder]/[cellname]_CELL/mat/
%   processOCV.mat = MAT file containing cellData structure
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.     

function processData = processOCV(cellspec,varargin)

  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addParameter('dv',2e-3,@(x)isscalar(x)&&x>0);
  parser.parse(cellspec,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  cellname = arg.cellspec.name;
  lockfile = arg.cellspec.ocv.lockfile;
  timestamp = arg.cellspec.timestamp__;
  
  outp.print('Started processOCV %s\n',cellname);
  
  % Load data from lockfile.
  lock = load(lockfile);
  
  % Iterate temperatures, flatten data structs.
  clear raw;
  for kt = length(lock.matData):-1:1 % !!! backwards so struct array doesn't grow
      raw(kt) = lock.matData(kt).dataOCV;
  end
  
  % Process full-cell data and estimate OCP-v-SOC
  outp.print('Processing %s ... \n',cellname);
  cal = calibrateSOC(raw,arg.dv);
  for k = 1:length(cal)
    cal(k).disZ = 1-cal(k).disZ;  % !! discharge from 100%
    cal(k).chgZ = 1-cal(k).chgZ;  % !! charge from 0%
  end
  if outp.text
    outp.info('  %-7s %-7s %-7s\n','TdegC','Q [Ah]','Eta [-]');
    for kt = 1:length(cal)
      outp.info('  %-7.3f %-7.3f %-7.3f\n', ...
          cal(kt).TdegC,cal(kt).QAh,cal(kt).eta);
    end
  end
  if outp.debug
      makePlots(cal,cellname);
  end
  
  % Save params to cellData struct.
  cellData = struct;
  cellData.const.TdegC = [cal.TdegC];
  cellData.const.Q = [cal.QAh];
  
  % Collect and save output data.
  processData.cellData = cellData;
  processData.raw = raw;
  processData.cal = cal;
  processData.name = 'FULL';  % full cell OCV
  processData.origin__ = 'processOCV';
  processData.arg__ = arg;
  processData.timestamp__ = timestamp;
  save(arg.cellspec.ocv.processfile,'-struct','processData');
  
  outp.print('Finished processOCV %s\n\n',cellname);
  
end
  
  
% Utility functions -------------------------------------------------------
  
%MAKEPLOTS Plot voltage-v-soc and differential capacity.
function makePlots(cal,cellname)
  
  col = cool(length(cal));
  lab1 = arrayfun( ...
      @(kt)sprintf('Dis'), ...
      1:length(cal), ...
      'UniformOutput',false);
  lab2 = arrayfun( ...
      @(kt)sprintf('Chg %.0f\\circC',cal(kt).TdegC), ...
      1:length(cal), ...
      'UniformOutput',false);
  
  figure;
  for kt = 1:length(cal)
    plot(cal(kt).disZ*100,cal(kt).disV,'-','Color',col(kt,:)); hold on;
  end
  for kt = 1:length(cal)
    plot(cal(kt).chgZ*100,cal(kt).chgV,':','Color',col(kt,:));
  end
  xlim([0 100]);
  xlabel('State of charge, z [%]');
  ylabel('Cell voltage [V]');
  title(sprintf('processOCV %s',cellname));
  legend(lab1{:},lab2{:},'Location','best','NumColumns',2);
  thesisFormat;
  
  figure;
  for kt = 1:length(cal)
    plot(cal(kt).disZ*100,cal(kt).disdZ,'-','Color',col(kt,:)); hold on;
  end
  for kt = 1:length(cal)
    plot(cal(kt).chgZ*100,cal(kt).chgdZ,':','Color',col(kt,:));
  end
  xlim([0 100]);
  xlabel('State of charge, z [%]');
  ylabel('Differential capacity [1/V]');
  title(sprintf('processOCV %s',cellname));
  legend(lab1{:},lab2{:},'Location','best','NumColumns',2);
  thesisFormat;
  figure;
  for kt = 1:length(cal)
    plot(cal(kt).disV,cal(kt).disdZ,'-','Color',col(kt,:)); hold on;
  end
  for kt = 1:length(cal)
    plot(cal(kt).chgV,cal(kt).chgdZ,':','Color',col(kt,:));
  end
  xlabel('Cell voltage [V]');
  ylabel('Differential capacity [1/V]');
  title(sprintf('processOCV %s',cellname));
  legend(lab1{:},lab2{:},'Location','best','NumColumns',2);
  thesisFormat;
  
end