% PROCESSOCP Preprocess raw OCP data from half cells. 
%
% * Compute total capacity (Q) and couloumbic efficiency (eta) @ each temp
% * Convert OCP-vs-time data to OCP-vs-SOC data @ each temp
% * Smooth OCP-vs-SOC curves and compute differential capacity @ each temp
%
% -- Usage --
% cellData = processOCP(cellspec)
% cellData = processOCP(...,'dvNEG',dvn)
% cellData = processOCP(...,'dvPOS',dvn)
%
% -- Input --
% cellspec = cell specification generated by labcell() function
% dvn      = voltage resolution for negative electrode voltage-v-soc 
%            and differential capacity computation [V]
% dvp      = voltage resolution for positive electrode voltage-v-soc
%            and differential capacity computation [V]
%
% -- Output --
% cellData = struct with the following fields:
% 
%   .neg.raw : raw data for negative electrode @ each temp (struct array)
%   .neg.cal : calibrated data for negative electrode @ each temp  (struct array)
%   .pos.raw : raw data for positive electrode @ each temp (struct array)
%   .pos.cal : calibrated data for positive electrode @ each temp (struct array)
%
% -- Output Files --
% [folder]/[cellname]_CELL/mat/
%   processOCP.mat = MAT file containing cellData structure
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function processData = processOCP(cellspec,varargin)

  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addParameter('dvNEG',2e-3,@(x)isscalar(x)&&x>0);
  parser.addParameter('dvPOS',5e-3,@(x)isscalar(x)&&x>0);
  parser.parse(cellspec,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  cellname = arg.cellspec.name;
  lockfile = arg.cellspec.ocp.lockfile;
  hcname = arg.cellspec.hcname;
  timestamp = arg.cellspec.timestamp__;
  
  outp.print('Started processOCP %s%s\n',cellname,hcname);
  
  % Load data from lockfile.
  lock = load(lockfile);
  
  % Iterate temperatures, flatten data structs.
  clear rawNEG rawPOS;
  for kt = length(lock.matData):-1:1 % !!! backwards so struct array doesn't grow
    rawNEG(kt) = lock.matData(kt).dataOCPn;
    rawPOS(kt) = lock.matData(kt).dataOCPp;
  end
  
  % NEG ---------------------------------------------------------------------
  % Process negative-electrode half-cell data and estimate OCP-v-SOC
  outp.print('Processing %s%s NEG ... \n',cellname,hcname);
  calNEG = calibrateSOC(rawNEG,arg.dvNEG);
  if outp.text
    outp.info('  %-7s %-7s %-7s\n','TdegC','Q [mAh]','Eta [-]');
    for kt = 1:length(calNEG)
      outp.info('  %-7.3f %-7.3f %-7.3f\n', ...
          calNEG(kt).TdegC,1000*calNEG(kt).QAh,calNEG(kt).eta);
    end
  end
  if outp.debug
    makePlots(calNEG,cellname,hcname,'Neg');
  end
  
  % POS ---------------------------------------------------------------------
  % Process positive-electrode half-cell data and estimate OCP-v-SOC
  outp.print('Processing %s%s POS ... \n',cellname,hcname);
  calPOS = calibrateSOC(rawPOS,arg.dvPOS);
  if outp.text
    outp.info('  %-7s %-7s %-7s\n','TdegC','Q [mAh]','Eta [-]');
    for kt = 1:length(calPOS)
      outp.info('  %-7.3f %-7.3f %-7.3f\n', ...
          calPOS(kt).TdegC,1000*calPOS(kt).QAh,calPOS(kt).eta);
    end
  end
  if outp.debug
    makePlots(calPOS,cellname,hcname,'Pos');
  end
  
  % Save params to cellData struct.
  cellData = struct;
  cellData.const.TdegC = [calNEG.TdegC];
  cellData.neg.Q = [calNEG.QAh];
  cellData.pos.Q = [calPOS.QAh];
  
  % Collect and save output data.
  processData.cellData = cellData;
  processData.neg.raw = rawNEG;
  processData.neg.cal = calNEG;
  processData.neg.name = 'NEG';
  processData.pos.raw = rawPOS;
  processData.pos.cal = calPOS;
  processData.pos.name = 'POS';
  processData.origin__ = 'processOCP';
  processData.arg__ = arg;
  processData.timestamp__ = timestamp;
  save(arg.cellspec.ocp.processfile,'-struct','processData');
  
  outp.print('Finished processOCP %s%s\n\n',cellname,hcname);  
end
  
  
% Utility functions -------------------------------------------------------

%MAKEPLOTS Plot voltage-v-soc and differential capacity.
function makePlots(cal,cellname,hcname,hc)
  
  col = cool(length(cal));
  lab1 = arrayfun( ...
      @(kt)sprintf('Dis'), ...
      1:length(cal), ...
      'UniformOutput',false);
  lab2 = arrayfun( ...
      @(kt)sprintf('Chg %.0f\\circC',cal(kt).TdegC), ...
      1:length(cal), ...
      'UniformOutput',false);
  
  figure;
  for kt = 1:length(cal)
    plot(cal(kt).disZ*100,cal(kt).disV,'-','Color',col(kt,:)); hold on;
  end
  for kt = 1:length(cal)
    plot(cal(kt).chgZ*100,cal(kt).chgV,':','Color',col(kt,:));
  end
  xlim([0 100]);
  xlabel('State of charge, z [%]');
  ylabel('Cell voltage [V]');
  title(sprintf('processOCP %s%s: %s',cellname,hcname,hc));
  legend(lab1{:},lab2{:},'Location','best','NumColumns',2);
  thesisFormat;
  
  figure;
  for kt = 1:length(cal)
    plot(cal(kt).disZ*100,cal(kt).disdZ,'-','Color',col(kt,:)); hold on;
  end
  for kt = 1:length(cal)
    plot(cal(kt).chgZ*100,cal(kt).chgdZ,':','Color',col(kt,:));
  end
  xlim([0 100]);
  xlabel('State of charge, z [%]');
  ylabel('Differential capacity [1/V]');
  title(sprintf('processOCP %s%s: Neg',cellname,hcname));
  legend(lab1{:},lab2{:},'Location','best','NumColumns',2);
  thesisFormat;
  figure;
  for kt = 1:length(cal)
    plot(cal(kt).disV,cal(kt).disdZ,'-','Color',col(kt,:)); hold on;
  end
  for kt = 1:length(cal)
    plot(cal(kt).chgV,cal(kt).chgdZ,':','Color',col(kt,:));
  end
  xlabel('Cell voltage [V]');
  ylabel('Differential capacity [1/V]');
  title(sprintf('processOCP %s%s: %s',cellname,hcname,hc));
  legend(lab1{:},lab2{:},'Location','best','NumColumns',2);
  thesisFormat;

end