%PROCESSEIS Preprocess EIS measurements, including:
%  - Convert OCP at each SOC setpoint into actual SOC by inverting OCP fn.
%  - Extrapolate impedance over high frequency by fitting inductance ECM.
%  - Run GDRT to obtain distribution of relaxation times (DRT) from 
%    impedance measurements. This enables us to remove the contribution
%    of cell inductance to impedance for TF model regression.
%
% -- Usage --
% cellData = processEIS(cellspec,config)
%
% -- Input --
% cellspec     = cell specification generated by the labcell() function
% config       = structure with the following configuration parameters:
%   .preprocessfn = Handle to function that pre-processes EIS measurements 
%                at a given temperature and SOC setpoint (OPTIONAL). 
%                Accepts the following parameters:
%                   Z      = complex impedance vector [Ohm]
%                   freq   = frequency vector [Hz]
%                   TdegC  = temperature [degC]
%                   socPct = SOC setpoint [%]
%                Returns Znew and freqNew, the preprocessed impedance
%                and frequency vectors.
%
% -- Output --
% cellData = struct with the following fields:
%
% -- Output Files --
% [folder]/[cellname]_CELL/mat/
%   processEIS.mat = MAT file containing cellData structure
%
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function processData = processEIS(cellspec,config,varargin)

  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addRequired('config',@(x)isstruct(x)&&isscalar(x));
  parser.addParameter('UseCached',true,@islogical);
  parser.parse(cellspec,config,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  cellname = arg.cellspec.name;
  lockfile = arg.cellspec.eis.lockfile;
  processfilename = arg.cellspec.eis.processfilename;
  processfile = arg.cellspec.eis.processfile;
  fitFilePulse = arg.cellspec.pls.fitfile;
  timestamp = arg.cellspec.timestamp__;
  
  outp.print('Started processEIS %s\n',cellname);
  
  % Load cellData from pulse file.
  dataPulse = load(fitFilePulse);
  cellData = dataPulse.cellData;
  
  % First, check for existing process file. If it exists, we'll load data 
  % from it directly to save some time.
  if arg.UseCached && isfile(processfile)
    % Lock file exists.
    processData = load(processfile);
    outp.info(['INFO: %s found. Using cached data.\n' ...
        '  Timestamp: %s \n' ...
        '  Note: Pass (''UseCached'',false) to force re-read\n'], ...
        processfilename, processData.timestamp__);
  else
    % Load data from lockfile.
    lock = load(lockfile);
    matData = lock.matData;
    
    % Process EIS data:
    % * Compute true SOC at each setpoint
    % * Apply user-defined preprocess function
    % * Compute GDRT
    clear eisVect;    % flat struct vector of impedance spectra
    cursor = 1;       % index into the vector above
    % -- Iterate temperatures --
    for kt = 1:length(matData)
      dataT = matData(kt);
      TdegC = dataT.TdegC;
      socSeries = dataT.socSeries;
      % -- Compute OCV-vs-SOC curve at this temperature --
      soc = linspace(0,1,1000);  % evalulate at 1000 discrete points
      thetaNEG = cellData.neg.theta0 + ...
          soc.*(cellData.neg.theta100 - cellData.neg.theta0);
      thetaPOS = cellData.pos.theta0 + ...
          soc.*(cellData.pos.theta100 - cellData.pos.theta0);
      UocpNEG = MSMR(cellData.neg).ocp('theta',thetaNEG,'TdegC',TdegC).Uocp;
      UocpPOS = MSMR(cellData.pos).ocp('theta',thetaPOS,'TdegC',TdegC).Uocp;
      UocvCELL = UocpPOS - UocpNEG;
      % -- Iterate SOC setpoints --
      for kz = 1:length(socSeries)
        dataZ = socSeries(kz);
        % -- Compute true SOC --
        ocv0 = mean(dataZ.ocv0.vcell);
        soc0 = fzero(@(zPct)linearinterp(soc,UocvCELL,zPct)-ocv0,0.5);
        soc0Pct = soc0*100;
        % -- Output to command window --
        outp.print( ...
            'Processing data at %5.2fdegC %5.2f%% SOC... ', ...
            TdegC,soc0Pct);
        % -- Preprocess impedance measurements --
        freqMeas = dataZ.freq;
        Zmeas = dataZ.Z;
        if isfield(arg.config,'preprocessfn')
          [ZPP,freqPP] = arg.config.preprocessfn( ...
              Zmeas,freqMeas,TdegC,soc0Pct);
        else
          ZPP = Zmeas;
          freqPP = freqMeas;
        end
        % -- Calculate DRT --
        % This step is slow...
        dataDRT = computeGDRT(freqPP,ZPP);
        % -- Store results --
        eis = struct;
        % Temperature and SOC setpoint.
        eis.TdegC = TdegC;
        eis.socPct = soc0Pct;
        % Measured impedance.
        eis.meas.freq = freqMeas;
        eis.meas.Z = Zmeas;
        % Preprocessed measured impedance.
        eis.pro.freq = freqPP;
        eis.pro.Z = ZPP;
        % Impedance ready for model regiression (RC components only).
        ind = min(freqMeas)<=dataDRT.freq & dataDRT.freq<=max(freqMeas);
        eis.reg.freq = dataDRT.freq(ind);
        eis.reg.Z = dataDRT.Zdrt_RC(ind);
        eisVect(cursor) = eis; %#ok<AGROW>
        cursor = cursor + 1;
        outp.print('done!\n');
        % -- Output plots (debug) --
        if outp.debug
          % Impedance measurement.
          figure;
          plot(real(Zmeas),-imag(Zmeas),'bo:'); hold on;
          plot(real(ZPP),-imag(ZPP),'ro:');
          plot(real(dataDRT.Zdrt_RC),-imag(dataDRT.Zdrt_RC),'mo:');
          setAxesNyquist( ...
              'xdata',mean(real(dataDRT.Zdrt_RC))*[0.5 2], ...
              'ydata',mean(-imag(dataDRT.Zdrt_RC))*[0.5 2] ...
          );
          xlabel('Re');
          ylabel('-Im');
          title(sprintf('%.2fdegC / %.2f%%SOC',TdegC,soc0Pct));
          legend( ...
              'Measured','Preprocessed','RC Only (DRT)', ...
              'Location','best' ...
          );
          thesisFormat;
          % RC DRT.
          figure;
          semilogx(dataDRT.tau,dataDRT.gamma_RC);
          xlabel('\tau [s]');
          ylabel('g_{RC}(\tau)');
          thesisFormat;
          % RL DRT.
          figure;
          semilogx(dataDRT.tau,dataDRT.gamma_RL);
          xlabel('\tau [s]');
          ylabel('g_{RL}(\tau)');
          thesisFormat;
        end
      end % for soc
    end % for T

    % Postprocess EIS data:
    % * Flatten struct array by temperature
    % -- Iterate temperatures --
    clear eisVectFlat;
    fieldsCollapse = {'meas','pro','reg'};
    Tvect = unique([eisVect.TdegC]);
    for kt = 1:length(Tvect)
      % Select all EIS measurements at current temperature.
      TdegC = Tvect(kt);
      indT = TdegC==[eisVect.TdegC];
      eisVectT = eisVect(indT);
      % Collect SOC setpoints at present temperature.
      socPct = sort([eisVectT.socPct]);
      nz = length(socPct);
      % -- Iterature EIS structure fields to collapse --
      tmp = struct;
      tmp.TdegC = TdegC;
      tmp.socPct = socPct;
      for kf = 1:length(fieldsCollapse)
        fieldname = fieldsCollapse{kf};
        % -- Iterate SOC setpoints --
        nf = length(eisVectT(1).(fieldname).freq);
        freq = zeros(nf,nz);
        Z = zeros(nf,nz);
        for kz = 1:nz
          % Find EIS measurements at this SOC.
          ind = [eisVectT.socPct] == socPct(kz);
          eis = eisVectT(ind);
          freq(:,kz) = eis.(fieldname).freq(:);
          Z(:,kz) = eis.(fieldname).Z(:);
        end % for SOC
        tmp.(fieldname).freq = freq;
        tmp.(fieldname).Z = Z;
      end % fieldname
      eisVectFlat(kt) = tmp; %#ok<AGROW>
    end % for T
    
    % Collect and save output data.
    processData = struct;
    processData.cellData = cellData;
    processData.eisVect = eisVectFlat;
    processData.origin__ = 'processEIS';
    processData.arg__ = arg;
    processData.timestamp__ = timestamp;
    save(arg.cellspec.eis.processfile,'-struct','processData');
  end % else
  
  % -- Plot EIS --
  if outp.plot
    Tvect = [processData.eisVect.TdegC];
    for kt = 1:length(Tvect)
      TdegC = Tvect(kt);
      eis = processData.eisVect(kt);
      socPct = eis.socPct;
      nz = length(socPct);
      lab = arrayfun( ...
          @(z)sprintf('%.2f%%',z),socPct, ...
          'UniformOutput',false);
      col = cool(nz);
      figure;
      subplot(131);
      for kz = 1:nz
        %freq = tmp.meas.freq(:,kz);
        Z = eis.meas.Z(:,kz);
        plot(real(Z),-imag(Z),'o:','Color',col(kz,:));
        hold on;
      end % for
      [~,indOUT] = rmoutliers(abs(eis.meas.Z));
      setAxesNyquist( ...
          'xdata',real(eis.meas.Z(~indOUT,:)), ...
          'ydata',-imag(eis.meas.Z(~indOUT,:)));
      xlabel('Re(Z)');
      ylabel('-Im(Z)');
      %legend(lab{:},'Location','best')
      title(sprintf('Measurement (%.2fdegC)',TdegC));
      subplot(132);
      for kz = 1:nz
        %freq = tmp.pro.freq(:,kz);
        Z = eis.pro.Z(:,kz);
        plot(real(Z),-imag(Z),'o:','Color',col(kz,:));
        hold on;
      end % for
      [~,indOUT] = rmoutliers(abs(eis.pro.Z));
      setAxesNyquist( ...
          'xdata',real(eis.pro.Z(~indOUT,:)), ...
          'ydata',-imag(eis.pro.Z(~indOUT,:)));
      xlabel('Re(Z)');
      ylabel('-Im(Z)');
      %legend(lab{:},'Location','best')
      title(sprintf('Preprocessed (%.2fdegC)',TdegC));
      subplot(133);
      for kz = 1:nz
        %freq = tmp.reg.freq(:,kz);
        Z = eis.reg.Z(:,kz);
        plot(real(Z),-imag(Z),'o:','Color',col(kz,:));
        hold on;
      end % for
      [~,indOUT] = rmoutliers(abs(eis.reg.Z));
      setAxesNyquist( ...
          'xdata',real(eis.reg.Z(~indOUT,:)), ...
          'ydata',-imag(eis.reg.Z(~indOUT,:)));
      xlabel('Re(Z)');
      ylabel('-Im(Z)');
      %legend(lab{:},'Location','best')
      title(sprintf('RC Only (%.2fdegC)',TdegC));
      thesisFormat;
    end % for T
  end % if
  
  outp.print('Finished processEIS %s\n\n',cellname);

end % processEIS()