% PROCESSDISCHG Preprocess raw dis/charge data from full cell. 
%
% * Compute total capacity (Q) and couloumbic efficiency (eta) @ each temp
% * Convert OCV-vs-time data to OCV-vs-SOC data @ each temp
% * Smooth OCV-vs-SOC curves and compute differential capacity @ each temp
%
% -- Usage --
% cellData = processDischg(cellspec)
% cellData = processDischg(...,'dv',dv)
%
% -- Input --
% cellspec = cell specification generated by labcell() function
% dv       = voltage resolution for voltage-v-soc and differential 
%            capacity computation [V]
%
% -- Output --
% cellData = struct with the following fields:
% 
%   raw : raw data @ each temp and C-rate (struct array)
%   cal : calibrated data @ each temp and C-rate (struct array)
%
% -- Output Files --
% [folder]/[cellname]_CELL/mat/
%   processDischg.mat = MAT file containing cellData structure
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.    

function processData = processDischg(cellspec,varargin)

  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addParameter('dv',2e-3,@(x)isscalar(x)&&x>0);
  parser.parse(cellspec,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  cellname = arg.cellspec.name;
  lockfile = arg.cellspec.dis.lockfile;
  fitFileOCV = arg.cellspec.ocv.fitfile;
  fitFileOCP = arg.cellspec.ocp.fitfile;
  timestamp = arg.cellspec.timestamp__;
  
  outp.print('Started processDischg %s\n',cellname);
  
  % Load data from lockfile.
  lock = load(lockfile);
  
  % Iterate temperatures and C-rates, flatten data structs.
  clear raw;
  cursor = 1;
  for k = 1:length(lock.matData)
    dataDis = lock.matData(k).dataDis;
    for kr = 1:length(dataDis)
      raw(cursor) = dataDis(kr); %#ok<AGROW>
      cursor = cursor + 1;
    end
  end
  
  % Process full-cell data and estimate OCV-v-SOC
  outp.print('Processing %s ... \n',cellname);
  cal = calibrateSOC(raw,arg.dv);
  for k = 1:length(cal)
    cal(k).disZ = 1-cal(k).disZ;  % !! discharge from 100%
    cal(k).chgZ = 1-cal(k).chgZ;  % !! charge from 0%
    % Copy cellID and cellName to calibrated data structure.
    cal(k).cellID = raw(k).cellID;
    cal(k).cellName = raw(k).cellName;
    % cal contains the true C-rate in `C_rate_actual`, but also keep
    % the original designated C-rate for human reference
    cal(k).C_rate = raw(k).C_rate;
  end
  if outp.text
    outp.info('  %-7s %-7s %-7s %-7s %-7s\n','Cell','TdegC','C-rate','Q [Ah]','Eta [-]');
    for k = 1:length(cal)
      outp.info('  %-7s %-7.3f %-7.3f %-7.3f %-7.3f\n', ...
          cal(k).cellID,cal(k).TdegC,cal(k).C_rate,cal(k).QAh,cal(k).eta);
    end
  end
  if outp.debug
    makePlots(cal,cellname);
  end
  
  % Load cellData from OCV lockfile.
  dataOCV = load(fitFileOCV);
  dataOCP = load(fitFileOCP);
  cellData = dataOCV.cellData;
  
  % Collect and save output data.
  % !! NOTE: raw has C-rate setpoint, cal has measured/true C-rate
  processData.cellData = cellData;
  processData.raw = raw;
  processData.cal = cal;
  processData.dataOCV = dataOCV;
  processData.dataOCP = dataOCP;
  processData.origin__ = 'processDischg';
  processData.arg__ = arg;
  processData.timestamp__ = timestamp;
  save(arg.cellspec.dis.processfile,'-struct','processData');
  
  outp.print('Finished processDischg %s\n\n',cellname);
  
  end
  
  
% Utility functions -------------------------------------------------------
  
% MAKEPLOTS Plot voltage-v-soc and differential capacity.
function makePlots(cal,cellname)
  
  col = cool(length(cal));
  lab1 = arrayfun( ...
      @(kt)sprintf('Dis'), ...
      1:length(cal), ...
      'UniformOutput',false);
  lab2 = arrayfun( ...
      @(kt)sprintf('Chg %.0f\\circC %.2fC',cal(kt).TdegC,cal(kt).C_rate), ...
      1:length(cal), ...
      'UniformOutput',false);
  
  figure;
  for kt = 1:length(cal)
    plot(cal(kt).disZ*100,cal(kt).disV,'-','Color',col(kt,:)); hold on;
  end
  for kt = 1:length(cal)
    plot(cal(kt).chgZ*100,cal(kt).chgV,':','Color',col(kt,:));
  end
  xlim([0 100]);
  xlabel('State of charge, z [%]');
  ylabel('Cell voltage [V]');
  title(sprintf('processDischg %s',cellname));
  legend(lab1{:},lab2{:},'Location','best','NumColumns',2);
  thesisFormat;
  
  figure;
  for kt = 1:length(cal)
    plot(cal(kt).disZ*100,cal(kt).disdZ,'-','Color',col(kt,:)); hold on;
  end
  for kt = 1:length(cal)
    plot(cal(kt).chgZ*100,cal(kt).chgdZ,':','Color',col(kt,:));
  end
  xlim([0 100]);
  xlabel('State of charge, z [%]');
  ylabel('Differential capacity [1/V]');
  title(sprintf('processDischg %s',cellname));
  legend(lab1{:},lab2{:},'Location','best','NumColumns',2);
  thesisFormat;
  figure;
  for kt = 1:length(cal)
    plot(cal(kt).disV,cal(kt).disdZ,'-','Color',col(kt,:)); hold on;
  end
  for kt = 1:length(cal)
    plot(cal(kt).chgV,cal(kt).chgdZ,':','Color',col(kt,:));
  end
  xlabel('Cell voltage [V]');
  ylabel('Differential capacity [1/V]');
  title(sprintf('processDischg %s',cellname));
  legend(lab1{:},lab2{:},'Location','best','NumColumns',2);
  thesisFormat;
end