% OUTP Singleton class for controlling output level of toolbox
%  functions.
%
% OUTP.quiet sets the toolbox output level to Quiet 
%   (no output except warnings, errors, and required).
% OUTP.normal sets the toolbox output level to Normal 
%   (text output only).
% OUTP.verbose sets the toolbox output level to Verbose 
%   (text output and plots).
%
% OUTP.debug('on') turns on debug output
% OUTP.debug('off') suppresses debug output
% state = OUTP.debug returns true if debug mode is on, false otherwise
%
% OUTP.text returns true if the the current level is Normal or Verbose,
%   and false otherwise
% OUTP.plot returns true if the the current level is Verbose, and false
%   otherwise
%
% OUTP.print delagates to fprintf, but only when level is Normal or
%   Verbose
% OUTP.info is the same, but prints text in an emphasized style
% OUTP.warning is the same, but prints text in the warning style
% OUTP.ln prints a newline
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

classdef outp

  enumeration
    Quiet, Normal, Verbose
  end

  properties (Constant)
    % Default output level.
    default = outp.Quiet
  end

  methods (Static)
    function out = level(in)
      % Define global output level and set default.
      persistent level;
      if isempty(level)
        level = outp.default;
      end

      if exist('in','var')
        % Set output level.
        if isa(in,'outp')
          % member of enumeration class
          level = in;
        else
          % string
          level = outp.(in);
        end
      else
        % Read output level.
        out = level;
      end % if
    end % level()

    function ret = debug(state)
      persistent debug;
      if isempty(debug)
        debug = false;
      end

      if exist('state','var')
        % set
        if strcmpi(state,'on')
          debug = true;
        else
          debug = false;
        end
      else
        % Read
        ret = debug;
      end
    end

    function out = charcount(in)
      % Set output character count for backspacing.
      persistent nchar;
      if isempty(nchar)
        nchar = 0;
      end

      if exist('in','var')
        % Set.
        nchar = in;
      else
        % Read.
        out = nchar;
      end % if
    end % level()


    function ret = req()
      ret = outp.level()==outp.Quiet   || ...
            outp.level()==outp.Normal  || ... 
            outp.level()==outp.Verbose;
    end
    function ret = warn()
      ret = outp.level()==outp.Quiet   || ...
            outp.level()==outp.Normal  || ... 
            outp.level()==outp.Verbose;
    end
    function ret = text()
      ret = outp.level()==outp.Normal  || ...
            outp.level()==outp.Verbose;
    end
    function ret = plot()
      ret = outp.level()==outp.Verbose;
    end

    function quiet()
      outp.level(outp.Quiet);
    end
    function normal()
      outp.level(outp.Normal);
    end
    function verbose()
      outp.level(outp.Verbose);
    end

    function print(varargin)
      nchar = outp.charcount();
      if outp.text, nchar = nchar + fprintf(varargin{:}); end
      outp.charcount(nchar);
    end
    function info(varargin)
      nchar = outp.charcount();
      if outp.text, nchar = nchar + fprintf(varargin{:}); end
      outp.charcount(nchar);
    end
    function warning(varargin)
      nchar = outp.charcount();
      if outp.warn
        nchar = nchar + cprintf('Errors','WARNING: '); 
        nchar = nchar + cprintf('Errors',varargin{:}); 
      end
      outp.charcount(nchar);
    end
    function required(varargin)
      nchar = outp.charcount();
      if outp.req
        nchar = nchar + cprintf('Errors',varargin{:}); 
      end
      outp.charcount(nchar);
    end
    function ln()
      nchar = outp.charcount();
      if outp.text, nchar = nchar + fprintf('\n'); end
      outp.charcount(nchar);
    end
  end % static methods
end