% MAKEMATFILEPULSE Load raw pulse data into MAT files.
% 
% NOTE: Supports Gamry data format only.
%
% WARNING: Datafile names use opposite sign convention for current!
%   i.e., I<0 denotes discharge current and I>0 charge current inside 
%   filenames. This script corrects the sign convention.
%
% -- Usage --
% out = MAKEMATFILEPULSE(cellspec)
% out = MAKEMATFILEPULSE(cellspec,'UseCached',cache)
%
% -- Input --
% cellspec = cell specification generated by labcell() function
% cache    = when true, skips reading raw files if cache file present
%            when false, force re-reading raw files even if cache present
%
% -- Output --
% out = structure with the following fields:
%
%   .matData : Vector of structs containing data saved to MAT files 
%     for each test temperature. Each struct contains these fields:
%       .TdegC     : Test temperature [degC].
%       .dataPulse : 
%
% -- Output Files --
% [folder]/[cellname]_CELL/mat/
%   { [cellname]_Pulse_[TdegC].mat } = MAT files w/ data @ each temp
%     [cellname]_Pulse.lock.mat      = cache file w/ all data
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function out = makeMATfilePulse(cellspec,varargin)
  
  % Parse input arguments.
  parser = inputParser;
  parser.addRequired('celldata',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addParameter('UseCached',true,@islogical);
  parser.addParameter('UnitNames',{'Master','Serf1','Serf2'});
  parser.addParameter('InitialSOCPct',100);
  parser.parse(cellspec,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  % Collect arguments.
  cellname = arg.celldata.name;
  cellfolder = arg.celldata.folder;
  lockfilename = arg.celldata.pls.lockfilename;
  lockfile = arg.celldata.pls.lockfile;
  timestamp = arg.celldata.timestamp__;
  unitnames = arg.UnitNames;
  
  outp.print('Started makeMATfilePulse %s\n',cellname);
  
  % First, check for lock file. If it exists, we'll load data from it
  % directly to save some time.
  if arg.UseCached && isfile(lockfile)
    % Lock file exists.

    out = load(lockfile);
    outp.info(['INFO: %s found. Using cached data.\n' ...
        '  Timestamp: %s \n' ...
        '  Note: Pass (''UseCached'',false) to force re-read\n'], ...
        lockfilename, out.timestamp__);
    outp.print('Finished makeMATfilePulse %s\n\n',cellname);
    return;
  end
  
  % Lock file not present - process raw data.
  datadir = fullfile(cellfolder,'raw','PULSE');
  outp.print('Reading PULSE data at %s ...\n',datadir);
  
  % Fetch top-level folders and files.
  % * One text file identifying the cell.
  % * One folder for each test temperature.
  [files,foldersTEMP] = getContents(datadir);
  filesTXT = files(endsWith({files.name},'.txt')); % select text files
  fileINFO = filesTXT(1);  % should only be one text file w/ test info
  [~,cellName,~] = fileparts(fileINFO.name);
  
  % Iterature temperatures.
  clear matData;
  for kt = 1:length(foldersTEMP)
    folderTEMP = foldersTEMP(kt);

    % Determine test temperature.
    stringTEMP = folderTEMP.name;
    TdegC = str2double(stringTEMP(2:end));
    if stringTEMP(1)=='N', TdegC = -TdegC; end

    % Fetch sub-folders - one folder for each SOC setpoint
    [~,foldersSOC] = getContents(fullfile(folderTEMP.folder,folderTEMP.name));
    
    % Iterature SOC setpoints.
    clear nodesSOC;
    for kz = 1:length(foldersSOC)
      folderSOC = foldersSOC(kz);
      
      % Determine SOC setpoint in percent.
      stringSOC = folderSOC.name;
      socPct = str2double(stringSOC(4:end));

      outp.print('Processing %.2fdegC / %d%% SOC data...\n',TdegC,socPct);

      % Fetch sub-files.
      files = getContents(fullfile(folderSOC.folder,folderSOC.name));
      fnames = {files.name};
      filesOCV = files(startsWith(fnames,'OCV')&~endsWith(fnames,'FINISH.DTA'));
      fileOCVEND = files(startsWith(fnames,'OCV')&endsWith(fnames,'FINISH.DTA'));
      fileDISCHG = files(startsWith(fnames,'PWRDISCHARGE'));
      filesPULSE = struct;
      for ku = 1:length(unitnames)
        unitname = unitnames{ku};
        filesPULSE.(unitname) = files(startsWith(fnames,unitname));
      end % for unit

      % Determine current magnitudes and number of pulse repeats.
      vectI = zeros(size(filesOCV));
      for kp = 1:length(filesOCV)
        fileOCV = filesOCV(kp);
        parts = strsplit(fileOCV.name,'_');
        stringI = parts{2};
        vectI(kp) = str2double(stringI(2:end));
        if stringI(1)=='N', vectI(kp) = -vectI(kp); end
      end
      Nrepeat = sum(vectI==vectI(1));
      vectI = unique(vectI);

      % PROCESS OCV AND DISCHARGE ---------------------------------------
      ocvend = struct;
      if ~isempty(fileOCVEND)
        filepath = fullfile(fileOCVEND.folder,fileOCVEND.name);
        dataOCVEND = loadGamryDTA(filepath);
        ocvend.time = dataOCVEND.tables.CURVE.T;
        ocvend.vcell = dataOCVEND.tables.CURVE.Vf;
      else
        outp.warning('WARNING: Missing %s\n',filepath);
      end

      dischg = struct;
      if ~isempty(fileDISCHG)
        filepath = fullfile(fileDISCHG.folder,fileDISCHG.name);
        dataDISCHG = loadGamryDTA(filepath);
        dischg.time = dataDISCHG.tables.CURVE.T;
        dischg.iapp = -dataDISCHG.tables.CURVE.Im; % !!! Gamry sign convention
        dischg.vcell = dataDISCHG.tables.CURVE.Vf;
      else
        outp.warning('WARNING: Missing %s\n',filepath);
      end
      % -----------------------------------------------------------------

      % PROCESS PULSE DATA ----------------------------------------------
      clear nodesI;
      % Iterate pulse magnitudes.
      for ki = 1:length(vectI)
        I = vectI(ki);
        stringI = sprintf('%d',abs(I));
        if I<0
          stringI = ['N' stringI]; %#ok<AGROW>
        else 
          stringI = ['P' stringI]; %#ok<AGROW>
        end
        outp.print('  I=%5dA ',I);

        % Iterate repeats.
        clear nodesR;
        for repeat = 1:Nrepeat
          suffix = sprintf('%s_%d.DTA',stringI,repeat);

          % Collect OCV measurement data.
          fileOCV = filesOCV(endsWith({filesOCV.name},suffix));
          dataOCV = loadGamryDTA(fullfile(fileOCV.folder,fileOCV.name));
          ocv = struct;
          ocv.time = dataOCV.tables.CURVE.T;
          ocv.vcell = dataOCV.tables.CURVE.Vf;

          % Collect pulse data from the units.
          time = []; iapp = []; vcell = [];
          for ku = 1:length(unitnames)
            unitname = unitnames{ku};
            tmpFiles = filesPULSE.(unitname);
            filePULSE = tmpFiles(endsWith({tmpFiles.name},suffix));
            if ~isempty(filePULSE)
              dataPULSE = loadGamryDTA(fullfile(filePULSE.folder,filePULSE.name));
              time =  [time   dataPULSE.tables.CURVE.T];  %#ok<AGROW>
              iapp =  [iapp  -dataPULSE.tables.CURVE.Im]; %#ok<AGROW>
              vcell = [vcell  dataPULSE.tables.CURVE.Vf]; %#ok<AGROW>
            else
              % Slave unit not active.
              time = [time nan(size(time,1),1)]; %#ok<AGROW>
              iapp = [iapp zeros(size(iapp,1),1)]; %#ok<AGROW>
              vcell = [vcell nan(size(vcell,1),1)]; %#ok<AGROW>
            end % if else
          end % for unit
          pulse = struct;
          pulse.time = time;
          pulse.iapp = iapp;
          pulse.vcell = vcell;

          % Collect output data.
          nodeR = struct;
          nodeR.cellName = cellName;
          nodeR.TdegC = TdegC;
          nodeR.socPct = socPct;
          nodeR.I = -I; % !! filenames use Gamry's sign convention
          nodeR.repeat = repeat;
          nodeR.ocv = ocv;
          nodeR.pulse = pulse;
          nodesR(repeat) = nodeR; %#ok<AGROW>
          outp.print('.');
        end % for repeat
        outp.ln;

        nodeI = struct;
        nodeI.cellName = cellName;
        nodeI.TdegC = TdegC;
        nodeI.socPct = socPct;
        nodeI.I = -I;  % !! filenames use Gamry's sign convention
        nodeI.repeatSeries = nodesR;
        nodesI(ki) = nodeI; %#ok<AGROW>
      end % for pulse magnitude
      % -----------------------------------------------------------------

      nodeSOC = struct;
      nodeSOC.cellName = cellName;
      nodeSOC.TdegC = TdegC;
      nodeSOC.socPct = socPct;
      nodeSOC.dischg = dischg;
      nodeSOC.ocvend = ocvend;
      nodeSOC.magnitudeSeries = nodesI;
      nodesSOC(kz) = nodeSOC; %#ok<AGROW>
    end % for SOC

    % Sort by SOC descending (in order of the discharge).
    soc = [nodesSOC.socPct];
    [~,ind] = sort(soc,'descend');
    nodesSOC = nodesSOC(ind);

    % Store data at this temperature.
    nodeT = struct;
    nodeT.cellName = cellName;
    nodeT.TdegC = TdegC;
    nodeT.soc0Pct = arg.InitialSOCPct;  % starting SOC
    nodeT.socSeries = nodesSOC;
    matData(kt) = nodeT; %#ok<AGROW>
  end % for temperature
  
  % Collect output data.
  out.matData = matData;
  out.origin__ = 'makeMATfilePulse';
  out.arg__ = arg;
  out.timestamp__ = timestamp;
  
  % Save to MAT files for each temperature.
  for data = out.matData
    TdegC = data.TdegC;
    if TdegC<0 
      tempPrefix = 'N';
    else 
      tempPrefix = 'P'; 
    end
    saveName = sprintf( ...
        '%s_Pulse_%s%02d.mat', ...
        cellname,tempPrefix,abs(TdegC));
    save(fullfile(cellfolder,'mat',saveName),'-struct','data');
  end
  
  % Save lock file.
  save(lockfile,'-struct','out');
  
  outp.print('Finished makeMATfilePulse %s\n\n',cellname);

end % makeMATfilePulse()

% GETCONTENTS
function [files, folders] = getContents(datadir)
  nodes = dir(datadir);
  nodes = nodes(~startsWith({nodes.name},{'.','~'})); % exclude system files
  isfolder = [nodes.isdir];
  folders = nodes(isfolder);
  files = nodes(~isfolder);
end