% MAKEMATFILEOCP Load raw OCV data from full cell into MAT files.
% 
% NOTE: Supports Arbin data format only.
%
% -- Usage --
% out = MAKEMATFILEOCV(cellspec)
% out = MAKEMATFILEOCV(cellspec,'UseCached',cache)
%
% -- Input --
% cellspec = cell specification generated by labcell() function
% cache    = when true, skips reading raw files if cache file present
%            when false, force re-reading raw files even if cache present
%
% -- Output --
% out = structure with the following fields:
%
%   .matData : Vector of structs containing data saved to MAT files 
%     for each test temperature. Each struct contains these fields:
%       .TdegC     : Test temperature [degC].
%       .dataOCV   : Structure of OCV data for full cell.
%
% -- Output Files --
% [folder]/[cellname]_CELL/mat/
%   { [cellname]_OCV_[TdegC].mat } = MAT files w/ data @ each temp
%     [cellname]_OCV.lock.mat      = cache file w/ all data
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function out = makeMATfileOCV(cellspec,varargin)

  % Parse input arguments.
  parser = inputParser;
  parser.addRequired('celldata',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addParameter('UseCached',true,@islogical);
  % -- These should never change
  parser.addParameter('ScriptNames', ...
      {'script1','script2','script3','script4'},@iscell);
  parser.addParameter('ExcelHeaderNames',{ ...
      'Test_Time(s)','Step_Index','Current(A)','Voltage(V)',...
      'Charge_Capacity(Ah)','Discharge_Capacity(Ah)','Aux_Temperature_1(C)'}, ...
      @iscell);
  parser.addParameter('ExcelHeaderCodeNames',{ ...
      'time','step','current','voltage','chgAh','disAh','temp'},@iscell);
  % --
  parser.parse(cellspec,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  % Collect arguments.
  cellname = arg.celldata.name;
  cellfolder = arg.celldata.folder;
  lockfilename = arg.celldata.ocv.lockfilename;
  lockfile = arg.celldata.ocv.lockfile;
  timestamp = arg.celldata.timestamp__;
  scripts = arg.ScriptNames;
  excelHeads = arg.ExcelHeaderNames;
  excelHeadsCodeNames = arg.ExcelHeaderCodeNames;
  
  outp.print('Started makeMATfileOCV %s\n',cellname);
  
  % First, check for lock file. If it exists, we'll load data from it
  % directly to save some time.
  if arg.UseCached && isfile(lockfile)
    % Lock file exists.

    out = load(lockfile);
    outp.info(['INFO: %s found. Using cached data.\n' ...
        '  Timestamp: %s \n' ...
        '  Note: Pass (''UseCached'',false) to force re-read\n'], ...
        lockfilename, out.timestamp__);
  else
    % Lock file not present - process raw data.

    tmpData = struct; % structure to store file data (post-processed below)
    ocvdir = fullfile(cellfolder,'raw','OCV');
    outp.print('Reading OCV data at %s ...\n',ocvdir);

    % Identify all excel prefix names in the cell folder.
    files = {dir(ocvdir).name}';
    vecIsDir = [dir(ocvdir).isdir];
    realFileInd = ~vecIsDir; 
    files = files(realFileInd);
    xlsxFiles = files(contains(files,'.xlsx'));
    if isempty(xlsxFiles)
      error(['Could''nt locate any Arbin data (xlsx) files at %s. ' ...
          'Cannot continue.'],ocvdir);
    end
    
    %--------------------------------------------------
    % Read Arbin data (excel files)
    %--------------------------------------------------
    % Identify test temperatures
    tempStr = cell(numel(xlsxFiles)/4,1); % 4 scripts
    testStr = tempStr;
    for i = 1:numel(tempStr)
      tempStr{i} = xlsxFiles{i*4}(end-10:end-8);
      testStr{i} = xlsxFiles{i*4}(1:end-8);
    end
    
    % Loop over all test temperatures
    clear temparray;
    cursor = 1;
    top = length(tempStr)*length(scripts);
    for theT = numel(tempStr):-1:1 % !!! backwards so struct array doesn't grow
      Tstr = tempStr{theT};
      if Tstr(1)=='N',TC=-str2double(Tstr(2:end));end
      if Tstr(1)=='P',TC=+str2double(Tstr(2:end));end
      ind = strfind(testStr{theT},'_');
      testname = testStr{theT}(ind(1)+1:ind(2)-1);
      
      % Store some info to structure
      dataOCV = [];
      dataOCV.name  = cellname;
      dataOCV.test  = testname;
      dataOCV.TdegC = TC;
      
      % Loop over the scripts and read data
      for k = 1:numel(scripts)
        filename = sprintf('%s_S%d.xlsx',testStr{theT},k);
        filepath = fullfile(ocvdir,filename);
        outp.print('  (%2d of %2d) %s ...',cursor,top,filename);
        data = readARBINxlsx(filepath,excelHeads,excelHeadsCodeNames);
        outp.print(' done!\n');
        dataOCV.(scripts{k}) = data;
        cursor = cursor + 1;
      end
      tmparray(theT) = dataOCV;
    end % temperature loop
    tmpData.data = tmparray;
    [~,srcnames,srcext] = fileparts(xlsxFiles);
    tmpData.srcfiles = fullfile('OCV',strcat(srcnames,srcext));
    
    % Collect temperatures. 
    temp = [tmpData.data.TdegC];
    numTemp = length(temp);
    
    % Flatten tmpData into a single struct vector, one for each temperature.
    clear matData;
    for kt = numTemp:-1:1 % !!! backwards so struct array doesn't grow
      tData = struct;  % data struct for this temperature
      tData.TdegC = temp(kt);
      tData.dataOCV = tmpData.data(kt);
      tData.origin__ = 'makeMATfileOCV';
      tData.arg__ = arg;
      tData.timestamp__ = timestamp;
      matData(kt) = tData;
    end

    % Collect output data.
    out.matData = matData;
    out.srcfiles = [tmpData.srcfiles];
    out.origin__ = 'makeMATfileOCV';
    out.arg__ = arg;
    out.timestamp__ = timestamp;

    % Save to MAT files for each temperature.
    for data = out.matData
      TdegC = data.TdegC;
      if TdegC<0 
        tempPrefix = 'N';
      else 
        tempPrefix = 'P'; 
      end
      saveName = sprintf( ...
        '%s_OCV_%s%02d.mat', ...
        cellname,tempPrefix,abs(TdegC));
      save(fullfile(cellfolder,'mat',saveName),'-struct','data');
    end

    % Save lock file.
    save(lockfile,'-struct','out');
  end % else
  
  outp.print('Finished makeMATfileOCV %s\n\n',cellname);
  
end
  
  
% Utility functions -------------------------------------------------------
  
% READARBINXLSX Reads the excel data sheets and save results to structure
%
% Inputs: filePath (where to find the excel file, full path)
%         xlsxHeaders (which columns to read, header names)
%         matFields (the corresponding field names to save)
%
% Output: dataOut (data structure with fields defined in 'matFileds')
function dataOut = readARBINxlsx(filePath,xlsxHeaders,matFields)
  
  if ~isfile(filePath),error('%s does not exist!',filePath);end
  if numel(xlsxHeaders)~=numel(matFields)
      error('%s and %s should match!',xlsxHeaders,matFields);
  end
  
  % Initialize empty storage
  sheets = sheetnames(filePath);
  for k = 1:numel(matFields),dataOut.(matFields{k}) = [];end
  
  % Loop over all sheets to extract data
  for theSheet = 1:numel(sheets)  % loop over all sheets
    if strcmp(sheets{theSheet},'Global_Info'),continue;end
    if strcmp(sheets{theSheet},'Info'), continue; end
    if strcmp(sheets{theSheet},'Channel_Chart'), continue; end
    [num,txt,~] = xlsread(filePath,sheets{theSheet}); % read the sheet
    
    % Loop over all excel header names
    for theHeader = 1:numel(xlsxHeaders)
      ind = strcmp(txt,xlsxHeaders{theHeader});
      colInd = find(ind);
      if ~isempty(colInd)
        dataOut.(matFields{theHeader}) = ...
            [dataOut.(matFields{theHeader});num(:, colInd)];
      end
    end
  end
end