% MAKEMATFILEEIS Load electrochemal impedance data into MAT files.
% 
% NOTE: Supports Gamry data format only.
%
% -- Usage --
% out = MAKEMATFILEEIS(cellspec)
% out = MAKEMATFILEEIS(cellspec,'UseCached',cache)
%
% -- Input --
% cellspec = cell specification generated by labcell() function
% cache    = when true, skips reading raw files if cache file present
%            when false, force re-reading raw files even if cache present
%
% -- Output --
% out = structure with the following fields:
%
%   .matData : Vector of structs containing data saved to MAT files 
%     for each test temperature. Each struct contains these fields:
%       .cellName  : Name of the cell.
%       .TdegC     : Test temperature [degC].
%       .soc0Pct   : Initial SOC of the cell before measurements [%].
%       .socSeries : Struct array of impedance data collected at each SOC.
%
% -- Output Files --
% [folder]/[cellname]_CELL/mat/
%   { [cellname]_EIS_[TdegC].mat } = MAT files w/ data @ each temp
%     [cellname]_EIS.lock.mat      = cache file w/ all data
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function out = makeMATfileEIS(cellspec,varargin)
  
  % Parse input arguments.
  parser = inputParser;
  parser.addRequired('celldata',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addParameter('UseCached',true,@islogical);
  parser.addParameter('InitialSOCPct',100);
  parser.parse(cellspec,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  % Collect arguments.
  cellname = arg.celldata.name;
  cellfolder = arg.celldata.folder;
  lockfilename = arg.celldata.eis.lockfilename;
  lockfile = arg.celldata.eis.lockfile;
  timestamp = arg.celldata.timestamp__;
  
  outp.print('Started makeMATfileEIS %s\n',cellname);
  
  % First, check for lock file. If it exists, we'll load data from it
  % directly to save some time.
  if arg.UseCached && isfile(lockfile)
    % Lock file exists.
    out = load(lockfile);
    outp.info(['INFO: %s found. Using cached data.\n' ...
        '  Timestamp: %s \n' ...
        '  Note: Pass (''UseCached'',false) to force re-read\n'], ...
        lockfilename, out.timestamp__);
    outp.print('Finished makeMATfileEIS %s\n\n',cellname);
    return;
  end
  
  % Lock file not present - process raw data.
  datadir = fullfile(cellfolder,'raw','EIS');
  outp.print('Reading EIS data at %s ...\n',datadir);
  
  % Fetch top-level folders and files.
  % * One text file identifying the cell.
  % * One folder for each test temperature.
  [files,foldersTEMP] = getContents(datadir);
  filesTXT = files(endsWith({files.name},'.txt')); % select text files
  fileINFO = filesTXT(1);  % should only be one text file w/ test info
  [~,cellName,~] = fileparts(fileINFO.name);
  
  % Iterature temperatures.
  clear matData;
  for kt = 1:length(foldersTEMP)
    folderTEMP = foldersTEMP(kt);

    % Determine test temperature.
    dnameTEMP = folderTEMP.name;
    parts = strsplit(dnameTEMP,'_');
    stringTEMP = parts{1};
    TdegC = str2double(stringTEMP(2:end));
    if stringTEMP(1)=='N', TdegC = -TdegC; end

    % Fetch sub-files.
    files = getContents(fullfile(folderTEMP.folder,folderTEMP.name));
    fnames = {files.name};
    filesOCV0 = files(startsWith(fnames,'OCP_BEFORE')&contains(fnames,'#'));
    filesEIS = files(startsWith(fnames,'EISGALV')|startsWith(fnames,'HYBRIDEIS'));

    % Fetch SOC indicies.
    indSOC = zeros(length(filesOCV0),1);
    socStrings = cell(length(filesOCV0),1);
    for kz = 1:length(filesOCV0)
      fileOCV0 = filesOCV0(kz);
      parts = strsplit(fileOCV0.name,'.');
      parts = strsplit(parts{1},'_');
      indexString = parts{end};
      socStrings{kz} = indexString;
      indSOC(kz) = str2double(indexString(2:end)); % remove leading '#'
    end % for
    [indSOC,indSort] = sort(indSOC);
    socStrings = socStrings(indSort);
    nsoc = length(indSOC);

    % Collect data.
    clear socSeries;
    for kz = nsoc:-1:1
      socInd = indSOC(kz);
      socStr = socStrings{kz};

      outp.print('Processing %.2fdegC / SOC%s data...\n',TdegC,socStr);

      tmp = struct;
      tmp.socInd = socInd;

      % Process OCV.
      % We'll use the OCV to determine the SOC of the cell in
      % processEIS.m!
      fileOCV0 = filesOCV0(endsWith({filesOCV0.name},[socStr '.DTA']));
      filepath = fullfile(fileOCV0.folder,fileOCV0.name);
      dataOCV0 = loadGamryDTA(filepath);
      tmp.ocv0.time = dataOCV0.tables.CURVE.T;
      tmp.ocv0.vcell = dataOCV0.tables.CURVE.Vf;

      % Process EIS
      % NOTE: multiple files per SOC indicate distinct frequency bands;
      % we combine these measurements into a single impedance spectrum.
      filesEISatSOC = filesEIS(endsWith({filesEIS.name},[socStr '.DTA']));
      bandCount = length(filesEISatSOC);
      % Adjoin impedance data from each band.
      freq = [];
      Z = [];
      for kb = bandCount:-1:1  % iterate bands
        fileEISatSOCatBAND = filesEISatSOC(kb);
        eis = loadGamryEIS( ...
            fullfile(fileEISatSOCatBAND.folder,fileEISatSOCatBAND.name));
        freq = [freq; eis.freq(:)]; %#ok<AGROW>
        Z = [Z; eis.Z(:)]; %#ok<AGROW>
      end
      [freq,indSort] = sort(freq);
      Z = Z(indSort);
      tmp.freq = freq;
      tmp.Z = Z;

      socSeries(kz) = tmp;
    end % for SOC

    % Store data at this temperature.
    nodeT = struct;
    nodeT.cellName = cellName;
    nodeT.TdegC = TdegC;
    nodeT.soc0Pct = arg.InitialSOCPct;  % starting SOC
    nodeT.socSeries = socSeries;
    matData(kt) = nodeT; %#ok<AGROW>
  end % for temperature
  
  % Collect output data.
  out.matData = matData;
  out.origin__ = 'makeMATfileEIS';
  out.arg__ = arg;
  out.timestamp__ = timestamp;
  
  % Save to MAT files for each temperature.
  for data = out.matData
    TdegC = data.TdegC;
    if TdegC<0 
      tempPrefix = 'N';
    else 
      tempPrefix = 'P'; 
    end
    saveName = sprintf( ...
      '%s_EIS_%s%02d.mat', ...
      cellname,tempPrefix,abs(TdegC));
    save(fullfile(cellfolder,'mat',saveName),'-struct','data');
  end
  
  % Save lock file.
  save(lockfile,'-struct','out');
  
  outp.print('Finished makeMATfileEIS %s\n\n',cellname);
  
end % makeMATfileEIS()
  
% GETCONTENTS
function [files, folders] = getContents(datadir)
  nodes = dir(datadir);
  nodes = nodes(~startsWith({nodes.name},{'.','~'})); % exclude system files
  isfolder = [nodes.isdir];
  folders = nodes(isfolder);
  files = nodes(~isfolder);
end