% LOADGAMRYEIS Extract EIS information from a Gamry .DTA file.
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function eis = loadGamryEIS(filename,varargin)

  % Number of harmonics appearing in the DTA file.
  nharm = 10;
  
  eis = loadGamryDTA(filename,varargin{:});
  ztab = eis.tables.ZCURVE;
  eis.freq = ztab.Freq;
  
  % Check for drift-corrected data - use it if present.
  if any(strcmpi("ZrealDrCor",ztab.Properties.VariableNames))
    eis.Z = ztab.ZrealDrCor+1j*ztab.ZimagDrCor;
  else
    eis.Z = ztab.Zreal+1j*ztab.Zimag;
  end
  
  % Check for harmonic-response data.
  if any(strcmpi("Vthd",ztab.Properties.VariableNames))
    eis.Vthd = ztab.Vthd;
    eis.Ithd = ztab.Ithd;
    Vh = zeros(length(eis.freq),nharm);
    Ih = Vh;
    for h = 1:nharm
      nameVR = sprintf('VHr%d',h); nameIR = sprintf('IHr%d',h);
      nameVI = sprintf('VHi%d',h); nameII = sprintf('IHi%d',h);
      Vh(:,h) = ztab.(nameVR)+1j*ztab.(nameVI);
      Ih(:,h) = ztab.(nameIR)+1j*ztab.(nameII);
    end
    eis.Vh = Vh;
    eis.Ih = Ih;
    % Note: Gamry current convention is (+) for charge current,
    % so we do not need a negative sign here.
    eis.Zhh = Vh./(Ih(:,1).^(1:nharm));
  end

end