% LABCELL Generate metadata for lab data collected for cell on disk.
%  Pass the output of this function to toolbox functions.
%
% -- Usage --
% cell = LABCELL(cellname)
% cell = LABCELL(cellname,folder)
% cell = LABCELL(...,'HalfCellName',hcname)
%
% -- Input --
% cellname = name of the cell (should be a folder called [cellname_CELL])
% folder   = folder containing the cell folder (default pwd)
% hcname   = name of half cells (default none [''])
%
% -- Output --
% cellspec  = struct of cell metadata
%
% -- Input Files --
% The following input directory structure is expected:
%
% [folder]
%   /[cellName]_CELL
%     /raw
%       /OCP_[hcname]Pos
%       /OCP_[hcname]Neg
%       /OCV
%       /DISCHG
%       /PULSE
%       /EIS
%
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function cell = labcell(cellName,varargin)

  % Parse input arguments.
  parser = inputParser;
  parser.addRequired('cellName',@(x)ischar(x)||isstring(x));
  parser.addOptional('folder',pwd(),@(x)isfolder(x));
  parser.addParameter('HalfCellName','',@(x)ischar(x)||isstring(x));
  parser.parse(cellName,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  cell.name = arg.cellName;
  cell.hcname = arg.HalfCellName;
  cell.folder = fullfile(arg.folder,[cellName '_CELL']);
  cell.ocp.lockfilename = sprintf('%s%s_OCP.lock.mat',arg.cellName,arg.HalfCellName);
  cell.ocp.lockfile = fullfile(cell.folder,'mat',cell.ocp.lockfilename);
  cell.ocp.hcdata = [
    struct( ...
        'folder',fullfile(cell.folder,'raw'), ...
        'name',['OCP_' cell.hcname 'Neg'], ...
        'codename','neg' ...
    ), ...
    struct( ...
        'folder',fullfile(cell.folder,'raw'), ...
        'name',['OCP_' cell.hcname 'Pos'], ...
        'codename','pos' ...
    ), ...
  ]; % struct array of half-cell data
  cell.ocp.processfilename = sprintf('processOCP%s.mat',arg.HalfCellName);
  cell.ocp.processfile = fullfile(cell.folder,'mat',cell.ocp.processfilename);
  cell.ocp.fitfilename = sprintf('fitOCP%s.mat',arg.HalfCellName);
  cell.ocp.fitfile = fullfile(cell.folder,'mat',cell.ocp.fitfilename);
  cell.ocv.lockfilename = sprintf('%s_OCV.lock.mat',arg.cellName);
  cell.ocv.lockfile = fullfile(cell.folder,'mat',cell.ocv.lockfilename);
  cell.ocv.processfilename = sprintf('processOCV.mat');
  cell.ocv.processfile = fullfile(cell.folder,'mat',cell.ocv.processfilename);
  cell.ocv.fitfilename = sprintf('fitOCV.mat');
  cell.ocv.fitfile = fullfile(cell.folder,'mat',cell.ocv.fitfilename);
  cell.dis.lockfilename = sprintf('%s_Dis.lock.mat',arg.cellName);
  cell.dis.lockfile = fullfile(cell.folder,'mat',cell.dis.lockfilename);
  cell.dis.processfilename = sprintf('processDischg.mat');
  cell.dis.processfile = fullfile(cell.folder,'mat',cell.dis.processfilename);
  cell.dis.fitfilename = sprintf('fitDischg.mat');
  cell.dis.fitfile = fullfile(cell.folder,'mat',cell.dis.fitfilename);
  cell.pls.lockfilename = sprintf('%s_Pulse.lock.mat',arg.cellName);
  cell.pls.lockfile = fullfile(cell.folder,'mat',cell.pls.lockfilename);
  cell.pls.processfilename = sprintf('processPulse.mat');
  cell.pls.processfile = fullfile(cell.folder,'mat',cell.pls.processfilename);
  cell.pls.fitfilename = sprintf('fitPulse.mat');
  cell.pls.fitfile = fullfile(cell.folder,'mat',cell.pls.fitfilename);
  cell.eis.lockfilename = sprintf('%s_EIS.lock.mat',arg.cellName);
  cell.eis.lockfile = fullfile(cell.folder,'mat',cell.eis.lockfilename);
  cell.eis.processfilename = sprintf('processEIS.mat');
  cell.eis.processfile = fullfile(cell.folder,'mat',cell.eis.processfilename);
  cell.eis.fitfilename = sprintf('fitEIS.mat');
  cell.eis.fitfile = fullfile(cell.folder,'mat',cell.eis.fitfilename);
  cell.pss.fitfilename = sprintf('fitPSS.mat');
  cell.pss.fitfile = fullfile(cell.folder,'mat',cell.pss.fitfilename);
  cell.origin__ = 'labcell';
  cell.arg__ = arg;
  cell.timestamp__ = datetime("now");

end