% GUESSMSMR Attempt to identify MSMR parameters from lab data over
%  relative lithiation.
%
% This works well only when all of the galleries are observable
% in the OCP curve. Manual fudging of this guess is required 
% for materials such as NMC.
%
% The reported X values should be scaled by (theta_max-theta_min).
% Manually adjusting the X is neccesary to ensure sum(X)=1 
% after scaling. In particular, consider stretching the X of 
% the galleries at the "edges" of the differential capacity 
% curves whose X may be larger than shown in collected data.
%
% -- Usage --
% [U0,X,omega,dv] = guessMSMR(cellspec,electrode,J)
%
% -- Input --
%  cellspec   = cell specification generated by labcell() function
%  electrode  = selects electrode, 'NEG' or 'POS'
%  J          = number of galleries (actual found may be fewer)
%
% -- Output --
%   U0, X, omega = vectors of MSMR parameters
%   dv           = the bin size used for the differential capacity
%                  computation
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function [U0, X, omega, dv] = guessMSMR(cellspec, electrode, J, varargin)
  
  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addRequired('electrode',@(x)any(strcmpi(x,{'NEG','POS'})));
  parser.addRequired('J',@(x)isscalar(x)&&x>0);
  parser.addParameter('dvInitial',1e-3,@(x)isscalar(x)&&x>0);
  parser.addParameter('dvIncrement',0.5e-3,@(x)isscalar(x)&&x>0);
  parser.addParameter('dvFinal',0.1,@(x)isscalar(x)&&x>0);
  parser.addParameter('vmin',[],@(x)isscalar(x));
  parser.addParameter('vmax',[],@(x)isscalar(x));
  parser.parse(cellspec,electrode,J,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  R = 8.3144598;      % Molar gas constant [J/mol K]
  F = 96485.3329;     % Faraday constant [C/mol]
  
  % Collect arguments.
  electrode = lower(arg.electrode);
  J = arg.J;
  dvInitial = arg.dvInitial;
  dvIncrement = arg.dvIncrement;
  dvFinal = arg.dvFinal;
  
  outp.print('Started guessMSMR %s %s\n',arg.cellspec.name,upper(electrode));
  
  % Load voltage-v-SOC curves closest to 25degC.
  cellData = load(arg.cellspec.processfile);
  electrodeData = cellData.(electrode);
  cal = electrodeData.cal; % calibrated voltage-v-SOC for dis/charge
  TdegC = [cal.TdegC];
  [~,ind] = min(abs(TdegC-25));
  cal = cal(ind);
  TdegC = TdegC(ind);
  
  % Compute OCP-v-SOC using simple voltage averaging of dis/charge curves.
  Z = linspace(0,1,1000)'; % common SOC vector
  disV = linearinterp(cal.disZ,cal.disV,Z);
  chgV = linearinterp(cal.chgZ,cal.chgV,Z);
  V = (disV+chgV)/2;
  if ~isempty(arg.vmin)
    ind = V>arg.vmin;
    Z = Z(ind);
    V = V(ind);
  end
  if ~isempty(arg.vmax)
    ind = V<arg.vmax;
    Z = Z(ind);
    V = V(ind);
  end
  
  % Compute differential capacity with progressively incresing
  % smoothing until we have less than maxJ galleries.
  for dv = dvInitial:dvIncrement:dvFinal
    [dzdv_V0, V0, dzdv_Z0, Z0] = smoothdiff2(Z, V, dv);
    [~, idxVPeaks] = findpeaks(dzdv_V0);
    [~, idxZValley] = findpeaks(-dzdv_Z0);
    outp.info('  %3d peaks / %3d valleys @ dv = %5.2fmV\n', ...
        length(idxVPeaks),length(idxZValley),dv*1000);
    if(length(idxVPeaks) <= J)
       break;
    end
  end
  if length(idxVPeaks) < J
    outp.warning(['Number of galleries found (%d) is less than ' ...
        'target number of galleries (%d).\n'],length(idxVPeaks),J);
  end
  
  % Determine location of peaks over voltage, sort descending.
  vPeaks = V0(idxVPeaks);
  dzdvPeaks = dzdv_V0(idxVPeaks);
  [vPeaks, vSortIdx] = sort(vPeaks,'descend');
  dzdvPeaks = dzdvPeaks(vSortIdx);
  
  % Determine locations of "valleys" over stiochometry, sort
  % ascending.
  zValley = sort(Z0(idxZValley));
  
  % Estimate MSMR model parameters.
  U0 = vPeaks;
  X = diff([0 zValley 1]);
  
  if length(X) ~= length(U0)
    omega = []; 
    outp.warning(['X-U0 mismatch. Could not estimate omega. ' ...
        'Incomplete MSMR estimate. ' ...
        'Try changing the number of target galleries J.\n']);
  else
    f = F/R/(TdegC+273.15);
    omega = f*X./dzdvPeaks/4;
  end
  
  % Do plotting.
  if outp.plot
    figure;
    plot(Z,V); hold on;
    if ~isempty(omega)
      % Compute and plot MSMR estimate.
      params.U0 = U0;
      params.X = X;
      params.omega = omega;
      params.theta0 = 0;
      params.theta100 = 100;
      ocpData = MSMR(params).ocp('TdegC',TdegC);
      plot(ocpData.theta,ocpData.Uocp,':');
    end
    xlabel('SOC');
    ylabel('Uocp');
    title('guessMSMR: Uocp-vs-SOC curve');
    if ~isempty(omega)
        legend('Lab','MSMR Guess');
    end
    thesisFormat;
    figure;
    plot(V0,dzdv_V0); hold on;
    xline(U0);
    xlabel('Uocp');
    ylabel('dZ/dU');
    title('guessMSMR: dZ-vs-Uocp curve');
    thesisFormat;
    figure;
    plot(Z0,dzdv_Z0); hold on;
    xlabel('SOC');
    ylabel('dZ/dU');
    title('guessMSMR: dZ-vs-SOC curve');
    thesisFormat;
  end
  
  outp.print('Finished guessMSMR %s %s\n',arg.cellspec.name,upper(electrode));

end

% SMOOTHDIFF2 Numerically differentiate noisy data using the 
%  "histogram counting" method.
% 
% Given vectors x and y, computes dxdy assuming regular spacing in the x
% samples. Specify the resolution (bin width), which controls the amount 
% of smoothing, using dy.
%
% The ideal (i.e. the underlying noise free) y vector should be monotonic
% (i.e. either increasing or decreasing over its entrire domain).
%
% Use the outputs dxdy_y and yout for viewing dxdy over y.
% Use the outputs dxdy_x and xout for viewing dxdy over x.
% Due to some bins containing zero samples (see explaination below),
% the lengths of dxdy_y,yout and dxdy_x,xout may differ.

% Determine histogram bin edges using the inputs y and dy.
% We force edges(1)=min(y) and edges(end)=max(y) by adjusting dy as needed.
% (dytrue is the actual dy value.)
function [dxdy_y, yout, dxdy_x, xout, dytrue] = smoothdiff2(x, y, dy)

  miny = min(y); maxy = max(y); deltay = maxy - miny;
  nbins = ceil(deltay/dy); dytrue = deltay / nbins;
  yEdges = linspace(miny, maxy, nbins+1);  % one more edge than bin
  
  % Count occurences of y in each bin. Use the result to approximate dx/dy. 
  ny = histcounts(y, yEdges);
  dx = ny*(max(x)-min(x))/length(x);  % Change in x across each bin, vector.
  dxdy_y = dx/dytrue;
  
  % Some bins may contain zero samples (i.e. ny=0, zero slope.) This is okay 
  % when viewing dxdy over y, but when viewing dxdy over x there will be 
  % more than one value of dxdy assigned to each x (i.e. duplicate entries 
  % in the xout vector due to some dx=0.) For this reason we remove sample 
  % points for which dx=0 in the dxdy vs x output.
  dxdy_x = dxdy_y(ny~=0);
  dx = dx(ny~=0);
  
  % Edges of the x-bins corresponding to the y-bins.
  % Order depends on whether y increases or decreases with x.
  if y(1) > y(end)
    xEdges = max(x) - [0 cumsum(dx)];
  else
    xEdges = min(x) + [0 cumsum(dx)];
  end
  
  % Derivative values correspond the center of the bins.
  yout = (yEdges(1:end-1)+yEdges(2:end))/2;
  xout = (xEdges(1:end-1)+xEdges(2:end))/2;

end