% GETPULSERESISTANCE Calculate pulse resistance of a full LIB cell.
%
% -- Usage --
% [R0,SOCPCT,IAPP] = getPulseResistance(param,socPct,iapp,TdegC)
%
% -- Input --
% param = struct of cell parameter values from evalSetpoint.m
% socPct = vector of SOC setpoints at which to evalulate pulse resistance
% iapp = vector of current magnitudes at which to evalulate pulse resistance
% TdegC = temperature at which to evalulate pulse resistance
%
% -- Output --
% R0 = matrix of pulse resistance values (dim1=SOC, dim2=iapp)
% SOCPCT = matrix of SOC in percent for use with surf.m
% IAPP = matrix of IAPP for use with surf.m

function [R0,SOCPCT,IAPP] = getPulseResistance(param,socPct,iapp,TdegC,ocp,uselinear,reltol)

  if ~exist("uselinear",'var')
    uselinear = false;
  end
  if ~exist('reltol','var')
    % Relative tolerance for determining when the contributions of
    % Rf and (Rct||Rdl) are negligible relative to one another.
    % (see odefun() defined below).
    reltol = 1e-5;  % two orders of magnitude below default BVP reltol (1e-3)
  end
  
  % Setup BVP solver options.
  % (A little faster with "Vectorize" turned on.)
  bvpOpts = bvpset('Stats','off','Vectorized','on','Nmax',100);
  
  % if isfield(param,'function')
  %   p = evalSetpoint(param,0,mean(socPct)/100,TdegC+273.15);
  % else
  %   p = param;
  % end
  p = param;
  
  % Define constants.
  F = 96485.3365;      % Faraday's constant [C mol^-1]
  R = 8.3144621;       % Universal gas constant [J mol^-1K^-1]
  T = TdegC+273.15;    % Temperature [K]
  f = F/(R*T);
  
  % Fetch OCP at each SOC setpoint (all vectors).
  % Fetch OCP at each SOC setpoint (all vectors).
  if ~exist('ocp','var')
    % Fetch OCP curves.
    thetaNEG = p.neg.theta0 + ...
        (socPct/100)*(p.neg.theta100-p.neg.theta0);
    thetaPOS = p.pos.theta0 + ... 
        (socPct/100)*(p.pos.theta100-p.pos.theta0);
    if isfield(p.neg,'U0')
      UocpNEG = MSMR(p.neg).ocp('theta',thetaNEG,'TdegC',TdegC).Uocp;
    elseif isfield(param,'function')
      UocpNEG = param.function.neg.Uocp(thetaNEG,TdegC+273.15);
    else
      error('Could not resolve Uocp(neg). Cannot continue.');
    end
    if isfield(p.pos,'U0')
      UocpPOS = MSMR(p.pos).ocp('theta',thetaPOS,'TdegC',TdegC).Uocp;
    elseif isfield(param,'function')
      UocpPOS = param.function.pos.Uocp(thetaPOS,TdegC+273.15);
    else
      error('Could not resolve Uocp(pos). Cannot continue.');
    end
  else
    thetaNEG = ocp.thetaNEG;
    UocpNEG = ocp.UocpNEG;
    thetaPOS = ocp.thetaPOS;
    UocpPOS = ocp.UocpPOS;
  end

  % If vector iapp given, make matrix by repeating for
  % each SOC setpoint.
  if isvector(iapp)
    iapp = repmat(iapp(:).',length(socPct),1);
  end
  
  % Fetch cell parameter values.
  sigman = p.neg.sigma;
  sigmap = p.pos.sigma;
  kappan = p.neg.kappa;
  kappas = p.sep.kappa;
  kappap = p.pos.kappa;
  Rdln = p.neg.Rdl;
  Rdlp = p.pos.Rdl;
  Rfn = p.neg.Rf;
  Rfp = p.pos.Rf;
  alphan = p.neg.alpha;
  alphap = p.pos.alpha;
  k0n = p.neg.k0;
  k0p = p.pos.k0;
  if length(k0n) > 1
    U0n = p.neg.U0;
    Wn = p.neg.omega;
    Xn = p.neg.X;
  end
  if length(k0p) > 1
    U0p = p.pos.U0;
    Wp = p.pos.omega;
    Xp = p.pos.X;
  end
  
  % Check for errors.
  if length(thetaNEG)~=length(socPct)||...
   length(UocpNEG)~=length(socPct)||...
   length(thetaPOS)~=length(socPct)||...
   length(UocpPOS)~=length(socPct)
    error(['Lengths of thetaNEG,UocpNEG,thetaPOS,UocpPOS do not match ' ...
        'length of SOC vector. Cannot continue.'])
  end % if
  if Rfn == 0 && Rdln == 0
    error('Rdl and Rf (neg) cannot both be zero');
  end % if
  if Rfp == 0 && Rdlp == 0
    error('Rdl and Rf (pos) cannot both be zero');
  end % if
  
  % Allocate memory for output.
  dv = zeros(length(socPct),size(iapp,2));  % predicted delta(Vcell)
  SOCPCT = zeros(size(dv));
  IAPP = zeros(size(dv));
  
  %------------------------------------------------------------
  % Compute R0 using bvp5c solver (solve for phise equation)
  %------------------------------------------------------------
  
  K1n = (1-alphan)*f;
  K2n = -alphan*f;
  K1p = (1-alphap)*f;
  K2p = -alphap*f;
  
  for kz = 1:length(socPct)
    thetan = thetaNEG(kz);
    thetap = thetaPOS(kz);
    
    if length(k0n) > 1 % Then, this is an MSMR-kinetics model
      Uocpn = UocpNEG(kz); 
      Uocpp = UocpPOS(kz); 
    
      % Compute fractions xj and i0 of each gallary (vectors)
      xn = Xn./(1+exp(f*(Uocpn-U0n)./Wn)); % sum(xjn) = thetan
      xp = Xp./(1+exp(f*(Uocpp-U0p)./Wp)); % sum(xjp) = thetap
      i0n = k0n.*((xn).^(Wn.*alphan)).*((Xn-xn).^(Wn.*(1-alphan)));
      i0p = k0p.*((xp).^(Wp.*alphap)).*((Xp-xp).^(Wp.*(1-alphap)));
    else % This is a standard Butler-Volmer-kinetics model
      i0n = k0n.*((1-thetan)^(1-alphan))*(thetan^alphan);
      i0p = k0p.*((1-thetap)^(1-alphap))*(thetap^alphap);
    end

    % 2024.05.13 | Added | WH
    %  Compute Rct, R(infty), and xi for each electrode to for use in 
    %  odefun() defined below, and also for use in generating initial
    %  solution for the BVP.
    Rctn = 1/f/sum(i0n);
    if isinf(Rdln)
      RinfNEG = Rfn+Rctn;
    else
      RinfNEG = Rfn+(Rctn*Rdln)/(Rctn+Rdln);
    end
    xiNEG = sqrt((1/sigman+1/kappan)/RinfNEG);
    Rctp = 1/f/sum(i0p);
    if isinf(Rdlp)
      RinfPOS = Rfp+Rctp;
    else
      RinfPOS = Rfp+(Rctp*Rdlp)/(Rctp+Rdlp);
    end
    xiPOS = sqrt((1/sigmap+1/kappap)/RinfPOS);

    % For use in solving the reaction flux equations for Ifdl inside
    % odefun() defined below.
    % 2024.05.15 | Edited | WH
    %  Recast these residual functions from Ifdl to eta to improve
    %  numerical conditioning. Formerly, fzero failed to find a solution
    %  when phise was large (e.g., >=1000) since very small changes in Ifdl
    %  lead to large overpotential (eta), making one of the exp() terms go
    %  out of range to +-Inf. Writing these in terms of eta at least moves
    %  the problematic difference outside of the exp() terms.
    %  NOTE: eta  = phise-Rf*Ifdl
    %        Ifdl = (phise-eta)/Rf
    negFn = @(etan,phise)(...
        sum(i0n.*(exp(K1n.*etan) - exp(K2n.*etan))) ... % If
        + etan/Rdln ... % Idl
        - (phise-etan)/Rfn ...
    );
    posFn = @(etap,phise)(...
        sum(i0p.*(exp(K1p.*etap) - exp(K2p.*etap))) ... % If
        + etap/Rdlp ... % Idl
        - (phise-etap)/Rfp ...
    );
    % The old functions:
    % negFn = @(Ifdl,phise)(...
    %     sum(i0n.*(exp(K1n.*(phise-Rfn*Ifdl)) - exp(K2n.*(phise-Rfn*Ifdl)))) ... % If
    %     + (phise-Rfn*Ifdl)/Rdln ... % Idl
    %     - Ifdl...
    %     );
    % posFn = @(Ifdl,phise)(...
    %     sum(i0p.*(exp(K1p.*(phise-Rfp*Ifdl)) - exp(K2p.*(phise-Rfp*Ifdl)))) ... % If
    %     + (phise-Rfp*Ifdl)/Rdlp ... % Idl
    %     - Ifdl ...
    %     );
    % 
    % negMean = @(x,in) x/Rdln+sum(i0n.*(exp(K1n.*x)-exp(K2n.*x)))-in; % x = etan
    % posMean = @(x,ip) x/Rdlp+sum(i0p.*(exp(K1p.*x)-exp(K2p.*x)))+ip; % x = etap

    for ki = 1:size(iapp,2)
      ik = iapp(kz,ki);
    
      % Generate initial values for BVP solver.
      % Old quadratic initialization:
      % if Rdln == 0
      %     % Special case.
      %     phisemean = Rfn*ik;
      % else
      %     phisemean = fzeroFaster(negMean,0,[],ik) + Rfn*ik;
      % end
      % if Rdlp == 0
      %     % Special case
      %     phisemeap = Rfp*ik;
      % else
      %     phisemeap = fzeroFaster(posMean,0,[],ik) + Rfp*ik;
      % end
      % bn = -ik/sigman; an = (+ik/kappan-bn)/2; cn = phisemean-an/3-bn/2;
      % bp = +ik/sigmap; ap = (-ik/kappap-bp)/2; cp = phisemeap-ap/3-bp/2;
      % yinit = @(x) [an*x.^2+bn*x+cn;2*an*x+bn;...
      %               ap*(1-x).^2+bp*(1-x)+cp; ap*(2*(1-x)-6)-bp];

      % 2024.05.15 | WH
      % Compute linear solution. This is the result when we
      % linearize the reaction flux equation w/r/t the overpotential
      % by Taylor series expansion. Good approximation when the
      % overpotential is small.
      % -- NEG --
      ylinFnNEG = @(z)(ik./xiNEG/sinh(xiNEG))*[
          cosh(xiNEG*(z-1))/sigman + cosh(xiNEG*z)/kappan;
          sinh(xiNEG*(z-1))*xiNEG/sigman + sinh(xiNEG*z)*xiNEG/kappan;
      ];
      % -- POS --
      ylinFnPOS = @(z)(-ik./xiPOS/sinh(xiPOS))*[
          cosh(xiPOS*(z-1))/sigmap + cosh(xiPOS*z)/kappap;
          sinh(xiPOS*(z-1))*xiPOS/sigmap + sinh(xiPOS*z)*xiPOS/kappap;
      ];
      % -- STACKED [NEG;POS] --
      ylinFn = @(x)[
          ylinFnNEG(x);    % z = x   in neg
          ylinFnPOS(1-x);  % z = 1-x in pos
      ];

      % 2024.05.15 | WH
      % First, check if we actually need to run the BVP solver.
      % Do this by checking the maximum overpotential observed for the
      % linear solution.
      xpos = linspace(0,1,100);
      ylin = ylinFn(xpos);
      phiseNEG = ylin(1,:);
      ifdlNEG = phiseNEG/RinfNEG;
      etaNEGmax = max(abs(phiseNEG - ifdlNEG*Rfn));
      phisePOS = ylin(3,:);
      ifdlPOS = phisePOS/RinfPOS;
      etaPOSmax = max(abs(phisePOS - ifdlPOS*Rfp));
      if uselinear || f*etaNEGmax/2 <= 0.01 && f*etaPOSmax/2 <= 0.01
        % Linear solution OK!
        % With alpha=1 or alpha=0 (worst cases), this limits the maximum
        % error in the linear approximation of the Butler-Volmer equation
        % to about 1%.
        % When alpha=0.5, the max. error is about 0.002%.
        ysol = ylin;
      else
        % Too nonlinear - use BVP solver.
        init.solver = 'bvpinit';
        init.x      = [0 0.5 1];
        init.y      = ylinFn(init.x);
        init.yinit  = ylinFn;
        lastwarn('', '');
        try
          sol5c = bvp5c(@odefun,@bcfun,init,bvpOpts); % bvp5c ODE solver
          ysol = sol5c.y;
        catch ME
          % 2024.05.14 | WH
          % Sometimes the BVP solver fails due to singular Jacobian.
          % In this case, we hope that the linear solution is good
          % enough. Usually this occurs for unusual values of the model
          % parameters, and doesn't cause issues.
          % outp.warning(['Could not solve BVP. Using linear solution ' ...
          %    'instead. Detail: [%s] %s\n'], ...
          %    ME.identifier,ME.message);
          ysol = ylin;
        end
        [~,warnId] = lastwarn();
        if(~isempty(warnId))
          % 2024.05.14 | WH
          % Sometimes we get a warning when BVP solver can't find a 
          % solution within the specified tolerance.
          % We'll hope the solution that it did find is good enough.
          % outp.warning('Issue solving BVP: %s\n', warnId);
        end % if
      end % if
      phisen = [ysol(1,1) ysol(1,end)]; % phisen(0) and phisen(1)
      phisep = [ysol(3,1) ysol(3,end)]; % phisep(2) and phisep(3)
    
      % ------------------------------------------------------
      %  Compute dv
      % ------------------------------------------------------
      dv(kz,ki) = -ik/(kappan+sigman) -ik/kappas -ik/(kappap+sigmap) ...
          - sigman/(kappan+sigman)*phisen(1,2) ... % phisen(xtilde=1)
          - kappan/(kappan+sigman)*phisen(1,1) ... % phisen(xtilde=0)
          + kappap/(kappap+sigmap)*phisep(1,2) ... % phisep(xtilde=3)
          + sigmap/(kappap+sigmap)*phisep(1,1);    % phisep(xtilde=2)
      SOCPCT(kz,ki) = socPct(kz);
      IAPP(kz,ki) = ik;
    end % for ki
  end % for kz
  
  % Compute R0
  R0 = -dv./iapp;

  % When iapp=0, R0 is 0/0 (NaN) as computed above.
  % Replace with limit of pulse resistance for zero current
  % (equivalent to high-frequency gain computed by tfXX functions).
  ind0 = iapp==0;
  if any(ind0,'all')
    [kkz, kki] = find(ind0);
    for kz = kkz(:).'
      if length(k0n) > 1 % Then, this is an MSMR-kinetics model
        Uocpn = UocpNEG(kz); 
        Uocpp = UocpPOS(kz); 
        % Compute fractions xj and i0 of each gallary (vectors)
        xn = Xn./(1+exp(f*(Uocpn-U0n)./Wn)); % sum(xjn) = thetan
        xp = Xp./(1+exp(f*(Uocpp-U0p)./Wp)); % sum(xjp) = thetap
        i0n = k0n.*((xn).^(Wn.*alphan)).*((Xn-xn).^(Wn.*(1-alphan)));
        i0p = k0p.*((xp).^(Wp.*alphap)).*((Xp-xp).^(Wp.*(1-alphap)));
      else % This is a standard Butler-Volmer-kinetics model
        thetan = thetaNEG(kz);
        thetap = thetaPOS(kz);
        i0n = k0n.*((1-thetan)^(1-alphan))*(thetan^alphan);
        i0p = k0p.*((1-thetap)^(1-alphap))*(thetap^alphap);
      end
      Rctn = 1/f/sum(i0n);
      if isinf(Rdln)
        RinfNEG = Rfn+Rctn;
      else
        RinfNEG = Rfn+(Rctn*Rdln)/(Rctn+Rdln);
      end
      xiNEG = sqrt((1/sigman+1/kappan)/RinfNEG);
      Rctp = 1/f/sum(i0p);
      if isinf(Rdlp)
        RinfPOS = Rfp+Rctp;
      else
        RinfPOS = Rfp+(Rctp*Rdlp)/(Rctp+Rdlp);
      end
      xiPOS = sqrt((1/sigmap+1/kappap)/RinfPOS);
      phiseFnNEG = @(z)(1./xiNEG/sinh(xiNEG))*...
          (cosh(xiNEG*(z-1))/sigman + cosh(xiNEG*z)/kappan);
      phiseFnPOS = @(z)(-1./xiPOS/sinh(xiPOS))*...
          (cosh(xiPOS*(z-1))/sigmap + cosh(xiPOS*z)/kappap);

      phisen0 = phiseFnNEG(0);
      phisen1 = phiseFnNEG(1);
      phisep2 = phiseFnPOS(1);
      phisep3 = phiseFnPOS(0);
      R00 = 1/(kappan+sigman) + 1/kappas + 1/(kappap+sigmap) ...
        + phisen1*sigman/(kappan+sigman) ...
        + phisen0*kappan/(kappan+sigman) ...
        - phisep3*kappap/(kappap+sigmap) ...
        - phisep2*sigmap/(kappap+sigmap);
      R0(kz,kki) = R00;
    end % for soc
  end % if
  
  % ----------------------------------------------------------------------
  % Define ODE function: "x" input is not used
  % y    = [phisen;phisen';phisep;phisep'] 
  % dydx = [phisen';phisen'';phisep';phisep'']
  % ----------------------------------------------------------------------
  function dydx = odefun(~,y)
    % Define dydx and compute phisen' and phisep'
    dydx = 0*y; dydx(1,:) = y(2,:); dydx(3,:) = y(4,:);
    
    % Compute for phisen'' first
    if Rfn < reltol*Rdln*Rctn/(Rdln+Rctn)          % 2024.05.13 | WH
      % Rf is negligible! (Rdl||Rct) determines Ifdl
      dydx(2,:) = (1/sigman+1/kappan)*(...
          sum(i0n.*(exp(K1n.*y(1,:))-exp(K2n.*y(1,:))))...
          +y(1,:)/Rdln);
    elseif Rdln*Rctn/(Rdln+Rctn) < reltol*Rfn      % 2024.05.13 | WH
      % (Rdl||Rct) is negligible! Rf determines Ifdl
      dydx(2,:) = (1/sigman+1/kappan)*y(1,:)/Rfn;
    else
      % 2024.05.13 | WH
      % Better initial guess reduces fzero failures;
      % choosing eta=0 means residual is phise/Rf, which is
      % at least finite and something fzero can work with!
      for k = 1:size(y,2)
        phise = y(1,k);
        try
          eta = fzeroFaster(negFn,0,[],phise);
        catch
          error(['Could not find feasible solution for ifdl(neg). ' ...
                'Cannot continue.']);
        end
        if isnan(eta)
          error(['Could not find feasible solution for ifdl(neg). ' ...
                'Cannot continue.']);
        end
        ifdl = (phise-eta)/Rfn;
        dydx(2,k) = (1/sigman+1/kappan)*ifdl;
      end
    end
    
    % Compute for phisep'' next
    if Rfp < reltol*Rdlp*Rctp/(Rdlp+Rctp)          % 2024.05.13 | WH
      % Rf is negligible! (Rdl||Rct) determines Ifdl
      dydx(4,:) = (1/sigmap+1/kappap)*(...
          sum(i0p.*(exp(K1p.*y(3,:))-exp(K2p.*y(3,:))))...
          +y(3,:)/Rdlp);
    elseif Rdln*Rctn/(Rdln+Rctn) < reltol*Rfp      % 2024.05.13 | WH
      % (Rdl||Rct) is negligible! Rf determines Ifdl
      dydx(4,:) = (1/sigmap+1/kappap)*y(3,:)/Rfp;
    else
      % 2024.05.13 | WH
      % Better initial guess reduces fzero failures;
      % choosing eta=0 means residual is phise/Rf, which is
      % at least finite and something fzero can work with!
      for k = 1:size(y,2)
        phise = y(3,k);
        try
          eta = fzeroFaster(posFn,0,[],phise);
        catch
          error(['Could not find feasible solution for ifdl(pos). ' ...
              'Cannot continue.']);
        end
        if isnan(eta)
          error(['Could not find feasible solution for ifdl(pos). ' ...
              'Cannot continue.']);
        end
        ifdl = (phise-eta)/Rfp;
        dydx(4,k) = (1/sigmap+1/kappap)*ifdl;
      end % for
    end % if
  end % odefun()
  
  % ------------------------------------------------------------------------
  % Set up boundary conditions
  % y = [phisen;phisen';phisep;phisep'];
  % yl is the left-side boundary, yr is the right-side boundary
  % ------------------------------------------------------------------------
  function res = bcfun(yl,yr)
    res = [yl(2)+ik/sigman;  % xtilde = 0
           yl(4)-ik/kappap;  % xtilde = 2
           yr(2)-ik/kappan;  % xtilde = 1
           yr(4)+ik/sigmap]; % xtilde = 3
  end

end % getPulseResistance()


% -------------------------------------------------------------------------
% Replace MATLAB's built-in FZERO command with this stripped-down version
% that removes a lot of error checking and tracing capabilities in
% preference for speed. Runs about twice as fast. See help on FZERO
% -------------------------------------------------------------------------
function b = fzeroFaster(FunFcn,x,options,varargin) %#ok<INUSL>
  % I always call with four inputs: anon function, initial guess, empty
  % options [], and a second argument to the anonymous function.
  % I always call with one output, so other outputs from FZERO removed.

  % Initialization
  fcount = 0;
  iter = 0;
  intervaliter = 0;
  tol = eps;

  % Put first feval in try catch
  try
    fx = FunFcn(x,varargin{:});
  catch ME
    if ~isempty(Ffcnstr)
      error('MATLAB:fzero:InvalidFunctionSupplied',...
          getString(message('MATLAB:optimfun:fzero:InvalidFunctionSupplied',...
          sprintf('%s ==> %s','function_handle',Ffcnstr),ME.message)));
    else
      error('MATLAB:fzero:InvalidFunctionSupplied',...
          getString(message('MATLAB:optimfun:fzero:InvalidFunctionSupplied',...
          'function_handle',ME.message)));
    end
  end
  fcount = fcount + 1;
  if fx == 0
    b = x;
    return
  elseif ~isfinite(fx) || ~isreal(fx)
    error('MATLAB:fzero:ValueAtInitGuessComplexOrNotFinite',...
        getString(message('MATLAB:optimfun:fzero:ValueAtInitGuessComplexOrNotFinite')));
  end

  if x ~= 0
    dx = x/50;
  else
    dx = 1/50;
  end

  % Find changes of sign.
  twosqrt = sqrt(2);
  a = x; fa = fx; b = x; fb = fx;

  while (fa > 0) == (fb > 0)
    intervaliter = intervaliter + 1;
    dx = twosqrt*dx;
    a = x - dx;  fa = FunFcn(a,varargin{:});
    fcount = fcount + 1;
    if ~isfinite(fa) || ~isreal(fa) || ~isfinite(a)
        b = NaN; return
    end

    if (fa > 0) ~= (fb > 0) % check for different sign
      break
    end

    b = x + dx;  fb = FunFcn(b,varargin{:});
    if ~isfinite(fb) || ~isreal(fb) || ~isfinite(b)
      b = NaN; return
    end
    fcount = fcount + 1;
  end % while

  fc = fb;
  while fb ~= 0 && a ~= b
    % Ensure that b is the best result so far, a is the previous
    % value of b, and c is on the opposite side of the zero from b.
    if (fb > 0) == (fc > 0)
      c = a;  fc = fa;
      d = b - a;  e = d;
    end
    if abs(fc) < abs(fb)
      a = b;    b = c;    c = a;
      fa = fb;  fb = fc;  fc = fa;
    end

    % Convergence test and possible exit
    m = 0.5*(c - b);
    toler = 2.0*tol*max(abs(b),1.0);
    if (abs(m) <= toler) || (fb == 0.0)
      break
    end

    % Choose bisection or interpolation
    if (abs(e) < toler) || (abs(fa) <= abs(fb))
      % Bisection
      d = m;  e = m;
    else
      % Interpolation
      s = fb/fa;
      if (a == c)
        % Linear interpolation
        p = 2.0*m*s;
        q = 1.0 - s;
      else
        % Inverse quadratic interpolation
        q = fa/fc;
        r = fb/fc;
        p = s*(2.0*m*q*(q - r) - (b - a)*(r - 1.0));
        q = (q - 1.0)*(r - 1.0)*(s - 1.0);
      end
      if p > 0
        q = -q;
      else
        p = -p;
      end
      % Is interpolated point acceptable
      if (2.0*p < 3.0*m*q - abs(toler*q)) && (p < abs(0.5*e*q))
        e = d;  d = p/q;
      else
        d = m;  e = m;
      end
    end % Interpolation

    % Next point
    a = b;
    fa = fb;
    if abs(d) > toler
      b = b + d;
    elseif b > c
      b = b - toler;
    else
      b = b + toler;
    end
    fb = FunFcn(b,varargin{:});
    fcount = fcount + 1;
    iter = iter + 1;
  end % Main loop
end