% FITPSS Regress the pseudo-steady-state (PSS) model to cell
%  discharge or charge measurements.
%
% Estimates the true value of psi and rescales kD, qen, qes, and qep
% accordingly. Due to cell-to-cell variation, we also incorporate Q (cell
% capacity) and R0 (tab resistance) into the optimization, though these
% variables are not returned. PSS model assumes symmetric reations 
% (alpha=0.5).
%
% Optimizer: fmincon. Best results with discharge data.
%
% -- Usage --
% outData = FITPSS(cellspec,configREG,direction)
% outData = FITPSS(cellspec,configREG,direction,narrowing)
% outData = FITPSS(...,'ocpRES',ocpres)
%
% -- Input --
% cellspec    = cell specification generated by labcell() function
% configREG   = structure of configuration options for the model 
%               regression (see fields below)
% direction   = 'dis' or 'chg' (regress to discharge or charge data)
% ocpres      = number of points to use to approximate OCP curves of the
%               electrodes
% narrowing   = string to select dataset (optional, see below)
%
% The fields of the configREG struct are:
%   .init  : Structure of initial values for: psi and QAh.
%   .lb    : Structure of lower bounds.
%   .ub    : Structure of upper bounds.
%   .pin   : Structure of parameter values to pin (i.e., to not
%               optimize over). OPTIONAL.
%
% The optional argument, `narrowing`, selects the dataset. It is a string 
% with any of the following three parts (separated by whitespace). All of 
% these are optional and may appear in any order. If multiple datasets
% match, you will be promted to select one dataset on the command window.
%  - TdegC : A string of the form 'XXdegC' where XX is the temperature in
%            in degrees Celcius. (e.g., '25degC' and '-15degC' are valid.)
%  - rate  : A string of the form 'XXC' where XX is the C-rate.
%            (e.g., '1C' and '0.5C' are valid but 'C/2' is not.)
%  - cellID: A string specifying the ID of the cell.
%            (e.g., 'FP41'.)
% Examples: '25degC 1C', '1C', 'FP41 0.5C' are all valid narrowing strings.
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function outData = fitPSS(cellspec,configREG,direction,varargin)
  
  istemp = @(x)(ischar(x)&&endsWith(x,'degC')&& ~isnan(str2double(x(1:end-4))));
  israte = @(x)(ischar(x)&&endsWith(x,'C')&&~isnan(str2double(x(1:end-1))));
  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addRequired('configREG',@(x)isscalar(x)&&isstruct(x));
  parser.addRequired('direction',@(x)ischar(x)&&any(strcmp(x,{'dis','chg'})));
  parser.addOptional('narrowing',[],@(x)ischar(x));
  parser.addParameter('ocpRES',1000,@(x)isnumeric(x)&&isscalar(x)&&x>1);
  parser.parse(cellspec,configREG,direction,varargin{:});
  arg = parser.Results; % struct of validated arguments
  
  % Collect cell metadata.
  cellname = arg.cellspec.name;
  disProcessfile = arg.cellspec.dis.processfile;
  eisFitfile = arg.cellspec.eis.fitfile;
  pssFitFile = arg.cellspec.pss.fitfile;
  timestamp = arg.cellspec.timestamp__;
  
  outp.print('Started fitPSS %s\n',cellname);
  
  % Load processed discharge data.
  processData = load(disProcessfile);
  cal = processData.cal;  % struct array of calibrated dischg data
  raw = processData.raw;  % struct array of raw dischg data
  
  % Load cell parameters from EIS regression.
  fitData = load(eisFitfile);
  cellData = fitData.cellData;
  modelspec = fitData.modelspec;  % needed for evalulating params at temp.
  
  % Select the dataset based on the narrowing parameters.
  if isempty(arg.narrowing)
    matches = 1:length(cal);
  else
    specs = split(arg.narrowing);

    % Determine narrowing parameters.
    TdegC = [];
    C_rate = [];
    cellID = [];
    for k = 1:length(specs)
      spec = specs{k};
      if istemp(spec)
        TdegC = str2double(spec(1:end-4));
      elseif israte(spec)
        C_rate = str2double(spec(1:end-1));
      else
        cellID = spec;
      end
    end % for

    % Search for matching datasets.
    clear matches;
    cursor = length(cal);
    for k = length(cal):-1:1
      if ~isempty(TdegC) && cal(k).TdegC ~= TdegC
        continue;
      end
      if ~isempty(C_rate) && cal(k).C_rate ~= C_rate
        continue;
      end
      if ~isempty(cellID) && ~strcmp(cal(k).cellID,cellID)
        continue;
      end

      % We have a match!
      matches(cursor) = k;
      cursor = cursor - 1;
    end
    matches = matches(cursor+1:end);  % delete unfilled elements
    if isempty(matches)
      error('Couldn''t find any tests for ''%s''. Cannot continue.', ...
          arg.narrowing);
    end
  end % if
  
  % If multiple matches, then ask the user to select which dataset to use.
  if length(matches) > 1
    N = length(matches);
    outp.required('Multiple matching datasets! Please disambiguate.\n');
    outp.required('  %-2s %-7s %-7s %-7s %-7s\n', ...
        '#','Cell','TdegC','C-rate','Q [Ah]');
    for k = 1:N
      dat = cal(matches(k));
      outp.required('  %-2d %-7s %-7.2f %-7.2f %-7.2f\n', ...
          k,dat.cellID,dat.TdegC,dat.C_rate,dat.QAh);
    end
    selector = -1;
    while (~isnumeric(selector) || ~isscalar(selector) || selector < 1 || selector > N)
      commandwindow; % bring focus to command window
      outp.required('Enter dataset number (1-%d) > ',N);
      selector = input('');
    end
    datasetInd = matches(selector);
  else
      datasetInd = matches(1);
  end
  
  dataset = cal(datasetInd);
  datasetRaw = raw(datasetInd);
  outp.info('INFO: Using %s %.2fdegC %.2fC %.2fAh dataset.\n', ...
      dataset.cellID,dataset.TdegC,dataset.C_rate,dataset.QAh);
  
  % Fetch previously-identified cell parameter values at
  % the dataset temperature.
  % Store all values in parameter struct `p`.
  p = optutil.evaltemp(cellData,modelspec,dataset.TdegC);
  p.TdegC = dataset.TdegC;           % test temperature
  p.C_rate = dataset.C_rate_actual;  % actual C-rate
  p.psi0 = p.const.psi;              % initial value for psi
  p.QAh0 = p.const.Q;                % initial value for capacity
  
  % Fetch configuration data.
  configREG = evalstrings(arg.configREG,p);
  p.ts = configREG.ts;
  if isfield(configREG,'pin')
    p.pin = configREG.pin;
  else
    p.pin = struct;
  end
  
  % Fetch OCP curves.
  % NEG - use regressed MSMR model (absolute OCP)
  thetaNEG = linspace(0.001,0.999,arg.ocpRES);
  ocpNEG = MSMR(p.neg).ocp('theta',thetaNEG,'TdegC',p.TdegC);
  p.neg.theta = thetaNEG;
  p.neg.Uocp = ocpNEG.Uocp;
  p.neg.dUocp = ocpNEG.dUocp;
  % POS - use regressed MSMR model (absolute OCP)
  thetaPOS = linspace(0.001,0.999,arg.ocpRES);
  ocpPOS = MSMR(p.pos).ocp('theta',thetaPOS,'TdegC',p.TdegC);
  p.pos.theta = thetaPOS;
  p.pos.Uocp = ocpPOS.Uocp;
  p.pos.dUocp = ocpPOS.dUocp;
  
  % Fetch time-domain measurements and resample with uniform sample interval.
  timeLAB = dataset.([arg.direction 'T']);
  iappLAB = dataset.([arg.direction 'It']);
  vcellLAB = dataset.([arg.direction 'Vt']);
  timeUniform = 0:p.ts:max(timeLAB);
  p.iappLAB = linearinterp(timeLAB,iappLAB,timeUniform);
  p.vcellLAB = linearinterp(timeLAB,vcellLAB,timeUniform);
  p.timeLAB = timeUniform;
  
  % Set initial SOC of the cell.
  % 1. Determine initial cell voltage.
  if strcmp(arg.direction,'chg')
      vcell0 = mean(datasetRaw.script3.voltage(datasetRaw.script3.step==1));
  else
      vcell0 = mean(datasetRaw.script1.voltage(datasetRaw.script1.step==1));
  end
  % 2. Determine initial SOC by inverting OCP-vs-SOC curve.
  soc = linspace(0,1,10000);
  thetan = p.neg.theta0 + soc*(p.neg.theta100 - p.neg.theta0);
  thetap = p.pos.theta0 + soc*(p.pos.theta100 - p.pos.theta0);
  Uocpn = linearinterp(p.neg.theta,p.neg.Uocp,thetan);
  Uocpp = linearinterp(p.pos.theta,p.pos.Uocp,thetap);
  Uocv = Uocpp - Uocpn;
  p.soc0 = linearinterp(Uocv,soc,vcell0);
  
  % Configure and run optimizer.
  init = stuff(configREG.init,p.pin);  % initial vector
  lb = stuff(configREG.lb,p.pin);      % lower bound vector
  ub = stuff(configREG.ub,p.pin);      % upper bound vector
  p.lb = configREG.lb;
  p.ub = configREG.ub;
  optionsFMINCON = optimoptions( ...
      @fmincon, ...
      'Display','off',...
      'MaxFunEvals',500,'MaxIter',1000,...
      'TolFun',1e-8,'TolX',1e-12,'TolCon',1e-12 ...
  );
  outp.info('Running PSS regression...\n');
  vect = fmincon(@(x)cost(x,p),init,[],[],[],[],lb,ub,[],optionsFMINCON);
  params = explode(vect,p.pin);
  outp.info('finished regression\n');
  
  % Update cell parameters.
  cellData.neg.qe = cellData.neg.qe*(params.psi/p.psi0);
  cellData.sep.qe = cellData.sep.qe*(params.psi/p.psi0);
  cellData.pos.qe = cellData.pos.qe*(params.psi/p.psi0);
  cellData.const.kD = cellData.const.kD*(params.psi/p.psi0);
  cellData.const.psi = params.psi;
  
  % Collect and save output data.
  outData = struct;
  outData.cellData = cellData;
  outData.params = params;
  outData.origin__ = 'fitPSS';
  outData.arg__ = arg;
  outData.timestamp__ = timestamp;
  save(pssFitFile,'-struct','outData');
  
  outp.print('Finished fitPSS %s\n',cellname);
  
end
  
  
% Utility functions -------------------------------------------------------

% COST Cost function for the model regression.
function J = cost(vect,arg)
  
  persistent printcount_inner printcount_outer minJ fevals fig ...
             line_vcellLAB line_vcellMOD;
  if isempty(minJ)
    minJ = Inf;
  end
  if isempty(printcount_inner)
    printcount_inner = 0;
  end
  if isempty(printcount_outer)
    printcount_outer = 0;
  end
  if isempty(fevals)
    fevals = 0;
  end
  if outp.plot && (isempty(fig) || ~ishandle(fig))
    minJ = Inf;
    printcount_inner = 0;
    printcount_outer = 0;
    fevals = 0;
    fig = figure( ...
        'Name','PSS Model Regression' ...
    );
    for k = 1:2
      line_vcellLAB = plot(NaN,NaN,'k:'); hold on;
      line_vcellMOD = plot(NaN,NaN,'m');
      xlabel('Time, $t$ [min]', ...
          'Interpreter','latex');
      ylabel('$v_\mathrm{cell}$ [$\mathrm{V}$]','Interpreter','latex');
      title('PSS Model Regression');
      legend('Lab','Model','Location','north');
      thesisFormat;
      drawnow;
    end
  end
  
  % Fetch optimization parameter values.
  params = explode(vect,arg.pin);
  arg.params = params;
  
  % Update model parameters, keeping the ratios qe/psi and kD/psi constant.
  arg.neg.qe = arg.neg.qe*(params.psi/arg.psi0);
  arg.sep.qe = arg.sep.qe*(params.psi/arg.psi0);
  arg.pos.qe = arg.pos.qe*(params.psi/arg.psi0);
  arg.const.kD = arg.const.kD*(params.psi/arg.psi0);
  arg.const.psi = params.psi;
  arg.const.Q = params.QAh;
  arg.neg.Uocp  = @(theta,~)linearinterp(arg.neg.theta,arg.neg.Uocp,theta);
  arg.pos.Uocp  = @(theta,~)linearinterp(arg.pos.theta,arg.pos.Uocp,theta);
  arg.neg.dUocp = @(theta,~)linearinterp(arg.neg.theta,arg.neg.dUocp,theta);
  arg.pos.dUocp = @(theta,~)linearinterp(arg.pos.theta,arg.pos.dUocp,theta);
  
  % Compute prediction of SPM without series resistance (-iapp*R0) term.
  [vcellMODWithoutR0, warnings] = runPSS( ...
      arg.iappLAB,arg.timeLAB,arg.soc0,arg.TdegC,arg);
  vcellLAB = arg.vcellLAB; % true (measured) cell voltage
  
  % Increment function-evalulation count.
  fevals = fevals + 1;
  
  % 2024.10.29 - Allow R0 to be neg or pos, since parameter estimates from
  %   previous model regressions may be incorrect, making the model appear
  %   to have more resistance.
  % Approximate series resistance (R0) by least squares linear regression.
  ind = vcellLAB>=2.95;
  R0 = (-arg.iappLAB(ind))\(vcellLAB(ind)-vcellMODWithoutR0(ind));
  %R0 = lsqnonneg(-arg.iappLAB(ind),vcellLAB(ind)-vcellMODWithoutR0(ind));
  
  % Estimate cell voltage with R0 term.
  vcellMOD = vcellMODWithoutR0 - arg.iappLAB*R0;
  
  % Compute the RMS error between the predicted and actual voltages.
  J = rms(vcellLAB - vcellMOD);
  
  if J < minJ
    minJ = J;

    % -- Update inner text output --

    % Delete previous iteration's output if present.
    if printcount_inner+printcount_outer > 0
      outp.info(repmat('\b',1,printcount_inner+printcount_outer));
      printcount_outer = 0;
    end

    % Start tracking the number of characters output
    outp.charcount(0);

    % Output regression results so far.
    suffPsi = '';
    if abs((params.psi - arg.lb.psi)/arg.lb.psi) < 0.05
      suffPsi = '(< 5% of lb)';
    end
    if abs((params.psi - arg.ub.psi)/arg.ub.psi) < 0.05
      suffPsi = '(< 5% of ub)';
    end
    suffQAh = '';
    if abs((params.QAh - arg.lb.QAh)/arg.lb.QAh) < 0.05
      suffQAh = '(< 5% of lb)';
    end
    if abs((params.QAh - arg.ub.QAh)/arg.ub.QAh) < 0.05
      suffQAh = '(< 5% of ub)';
    end
    
    outp.info('  RMSE [mV]    : %-20.3f\n',1000*J);
    outp.info('  psi  [V.K-1] : %-20.3e %s\n',params.psi,suffPsi);
    outp.info('  psi0 [V.K-1] : %-20.3e\n',arg.psi0);
    outp.info('  Q    [A.h]   : %-20.3g %s\n',params.QAh,suffQAh);
    outp.info('  Q0   [A.h]   : %-20.3g\n',arg.QAh0);
    outp.info('  R0   [V.A-1] : %-20.3g\n',R0);
    if ~isempty(warnings.thetae)
      outp.print('  Warning: %s\n',warnings.thetae);
    end
    if ~isempty(warnings.thetass)
      outp.print('  Warning: %s\n',warnings.thetass);
    end

    % Save number of chars output so we can backspace next iteration  
    printcount_inner = outp.charcount();

    % -- Update plot --
    
    if outp.plot && ishandle(fig)
      set(line_vcellLAB,'XData',arg.timeLAB/60,'Ydata',vcellLAB);
      set(line_vcellMOD,'XData',arg.timeLAB/60,'Ydata',vcellMOD);
      drawnow;
    end
  end
  
  % -- Update outer text output --
  
  % Delete previous iteration's output if present.
  if printcount_outer > 0
    outp.info(repmat('\b',1,printcount_outer));
  end
  
  % Start tracking the number of characters output
  outp.charcount(0);
  
  % Output regression results so far.
  if ~isempty(warnings.thetae) || ~isempty(warnings.thetass)
    suffix = '(!)';
  else
    suffix = '';
  end
  outp.info('  f-evals      : %-d %s\n',fevals,suffix);
  
  % Save number of chars output so we can backspace next iteration  
  printcount_outer = outp.charcount();
  
end
  
% RUNPSS Simulate voltage prediction of the pseudo-steady-state (PSS) model.
% !! The output voltage (vcell) does not include the series resistance 
%    (-iapp*Rc) term. You must add this term manually after calling 
%    this function.
function [vcell, warn] = runPSS(iappVect,timeVect,soc0,TdegC,cellParams)
  
  warn.thetae = '';
  warn.thetass = '';
  
  % Assign constants.
  F = 96485.3365;           % Faraday's constant [C mol^-1]
  R = 8.3144621;            % Universal gas constant [J mol^-1K^-1]
  T = TdegC+273.15;         % Temperature [K]
  ts = timeVect(2)-timeVect(1);     % Sampling interval [s].
  
  % Collect model parameter values.
  th0n          = cellParams.neg.theta0;
  th0p          = cellParams.pos.theta0;
  th100n        = cellParams.neg.theta100;
  th100p        = cellParams.pos.theta100;
  dthn          = th100n - th0n;
  dthp          = th100p - th0p;
  Un            = cellParams.neg.Uocp;
  Up            = cellParams.pos.Uocp;
  dUn           = cellParams.neg.dUocp;
  dUp           = cellParams.pos.dUocp;
  Dsrefn        = cellParams.neg.Dsref;
  Dsrefp        = cellParams.pos.Dsref;
  kn            = cellParams.neg.kappa;
  ks            = cellParams.sep.kappa;
  kp            = cellParams.pos.kappa;
  sn            = cellParams.neg.sigma;
  sp            = cellParams.pos.sigma;
  Cdln          = cellParams.neg.Cdl;
  Cdlp          = cellParams.pos.Cdl;
  k0n           = cellParams.neg.k0;
  k0p           = cellParams.pos.k0;
  Rfn           = cellParams.neg.Rf;
  Rfp           = cellParams.pos.Rf;
  QAh           = cellParams.const.Q;
  psi           = cellParams.const.psi;
  qen           = cellParams.neg.qe;
  qes           = cellParams.sep.qe;
  qep           = cellParams.pos.qe;
  kD            = cellParams.const.kD;
  U0n           = cellParams.neg.U0;
  Xn            = cellParams.neg.X;
  omegan        = cellParams.neg.omega;
  U0p           = cellParams.pos.U0;
  Xp            = cellParams.pos.X;
  omegap        = cellParams.pos.omega;
  
  % Compute initial stoichiometries of the electrodes (relative for POS).
  thetaNEG0 = th0n + soc0*dthn;
  thetaPOS0 = th0p + soc0*dthp;
  
  % Compute constant current for discharge.
  iapp = mean(iappVect);
  
  % Initialize model --------------------------------------------------------
  
  f = F/R/T;
  
  % Average faradaic plus double-layer current.
  ifdl_avg_n = +iapp;
  ifdl_avg_p = -iapp;
  
  % Average surface stoichiometries.
  thetass_avg_n = thetaNEG0;
  thetass_avg_p = thetaPOS0;
  
  % Surface stoichiometries at the current-collectors.
  thetass0 = thetaNEG0;
  thetass3 = thetaPOS0;
  
  % Three-parameter approximation for average thetass:
  % - Volume-averaged stoichiometry.
  theta_bar_n = thetaNEG0;
  theta_bar_p = thetaPOS0;
  % - Volume-averaged flux.
  q_n = 0;
  q_p = 0;
  
  % Electrolyte stoichiometry at x=0,3
  % (equal to TF dc gain values multiplied by iapp plus 1.0 bias value).
  thetae0 = ...
      1.0 + ...
      iapp * ...
      (qen/6/kn + qes*(1/kn + 1/ks)/2 + qep*(1/kn/2 + 1/ks + 1/kp/3)) ...
      /(qen + qes + qep)/psi/T;
  thetae1 = thetae0 - iapp/kn/2/psi/T;
  thetae3 = thetae1 - iapp*(1/ks + 1/kp/2)/psi/T;
  
  % 2024.10.31 - Check reasonable thetae values.
  %   Values may go out of range for very small psi.
  if abs(thetae0 - 1) > 0.99 || abs(thetae3 - 1) > 0.99
      warn.thetae = sprintf(['thetae0=%.3f and thetae3=%.3f. ' ...
          'Try increasing lb on psi.'],thetae0,thetae3);
  end
  thetae0 = min(1.99, thetae0);
  thetae0 = max(0.01, thetae0);
  thetae3 = min(1.99, thetae3);
  thetae3 = max(0.01, thetae3);
  
  % Debiased electrolyte potential at x=3 
  % (phie3_tilde = phie3 - phie0 where phie0 = -phise0)
  % (equal to TF dc gain value multiplied by iapp).
  phie3_tilde = -iapp*(1 - kD/psi)*(1/kn/2 + 1/ks + 1/kp/2);
  
  % Pre-allocate output vector
  vcell = 0*iappVect;
  
  % Run simulation --------------------------------------------------------
  
  for index = 1:length(timeVect)
    % Baker-Verbrugge solid diffusivity at average stoichiometry.
    Dsn = f*Dsrefn*(thetass_avg_n - 1)*thetass_avg_n*dUn(thetass_avg_n);
    Dsp = f*Dsrefp*(thetass_avg_p - 1)*thetass_avg_p*dUp(thetass_avg_p);
    
    % The following code runs except in the first iteration to compute
    % the solid-surface stoichiometry at locations x=0 and x=3
    % (in the first iteration, we know thetass0 and thetass3 are equal 
    % to initial stoichiometries, and thus we do not compute them).
    if index > 1
      % Solid-surface stoichiometry approximation.
      theta_hat_ss_n = thetass_avg_n - abs(dthn)/QAh/3600*ts*if_avg_n;
      theta_hat_ss_p = thetass_avg_p - abs(dthp)/QAh/3600*ts*if_avg_p;

      % Faradaic current approximation.
      if_hat_n = ...
          ifdl_avg_n ...
          *3600*QAh ...
          ./(3600*QAh - Cdln*abs(dthn)*dUn(theta_hat_ss_n));
      if_hat_p = ...
          ifdl_avg_p ...
          *3600*QAh ...
          ./(3600*QAh - Cdlp*abs(dthp)*dUp(theta_hat_ss_p));

      % Three-parameter approximation for average thetass:
      % - Volume-averaged stoichiometry
      theta_bar_n = theta_bar_n ...
          - abs(dthn)*ts/2/QAh/3600*(if_hat_n + if_avg_n);
      theta_bar_p = theta_bar_p ... 
          - abs(dthp)*ts/2/QAh/3600*(if_hat_p + if_avg_p);
      % - Volume-averaged flux
      expn = exp(-30*Dsn*ts);
      expp = exp(-30*Dsp*ts);
      q_n = expn*q_n ... 
          + abs(dthn)/120/QAh/Dsn...
          *(...
            (1-expn)/ts*(if_hat_n - if_avg_n)...
            - 30*Dsn*(if_hat_n - expn*if_avg_n)...
          );
      q_p = expp*q_p ... 
          + abs(dthp)/120/QAh/Dsp...
          *(...
            (1-expp)/ts*(if_hat_p - if_avg_p) ...
            - 30*Dsp*(if_hat_p - expp*if_avg_p)...
          );
      % - Average solid-surface stoichiometry
      thetass_avg_n = theta_bar_n + 8/35*q_n ... 
          - abs(dthn)/105/3600/QAh/Dsn*if_hat_n;
      thetass_avg_p = theta_bar_p + 8/35*q_p ... 
          - abs(dthp)/105/3600/QAh/Dsp*if_hat_p;

      % 2024.10.29 - Clamp thetass_avg between 0 and 1.
      % Since we are optimizing over QAh, there may be error
      % near the end of the discharge profile when QAh is lower
      % than the true value.
      if(...
          thetass_avg_n < 0 || thetass_avg_n > 1 || ...
          thetass_avg_p < 0 || thetass_avg_p > 1 ...
      )
          warn.thetass = sprintf( ...
              'thetass_avg_n=%.5f thetass_avg_p=%.5f', ...
              thetass_avg_n,thetass_avg_p);
      end
      thetass_avg_n = max(0, thetass_avg_n);
      thetass_avg_n = min(1, thetass_avg_n);
      thetass_avg_p = max(0, thetass_avg_p);
      thetass_avg_p = min(1, thetass_avg_p);
      
      % Positional correction to average solid-surface stoichiometry.
      thetass0 = thetass_avg_n + ... 
          iapp*(1/3/sn - (1-kD/psi)/6/kn)/dUn(thetass_avg_n);
      thetass3 = thetass_avg_p - ...
          iapp*(1/3/sp - (1-kD/psi)/6/kp)/dUp(thetass_avg_p);
    end
    
    % Corrected faradaic flux at locations:
    % - x=[avg_n; 0]
    if_n = ...
        ifdl_avg_n ...
        *3600*QAh ...
        ./(3600*QAh - Cdln*abs(dthn)*dUn([thetass_avg_n; thetass0]));
    if_avg_n = if_n(1);
    if0 = if_n(2);
    % - x=[avg_p; 3]
    if_p = ...
        ifdl_avg_p ...
        *3600*QAh ...
        ./(3600*QAh - Cdlp*abs(dthp)*dUp([thetass_avg_p; thetass3]));
    if_avg_p = if_p(1);
    if3 = if_p(2);

    % Butler-volmer inverse.
    % 2024.10.31 - Use MSMR exchange currents.
    %  Old expressions:
    %   i0n = sum(k0n*(thetae0).^0.5.*thetass0.^0.5.*(1-thetass0).^0.5);
    %   i0p = sum(k0p*(thetae3).^0.5.*thetass3.^0.5.*(1-thetass3).^0.5);
    Ussn0 = Un(thetass0);
    gjssn = exp(f*(Ussn0-U0n)./omegan);
    xjssn = Xn./(1+gjssn);
    i0jn = k0n.*xjssn.^(omegan/2).*(Xn-xjssn).^(omegan/2).*thetae0.^0.5;
    i0n = sum(i0jn,1);
    Ussp3 = Up(thetass3);
    gjssp = exp(f*(Ussp3-U0p)./omegap);
    xjssp = Xp./(1+gjssp);
    i0jp = k0p.*xjssp.^(omegap/2).*(Xp-xjssp).^(omegap/2).*thetae3.^0.5;
    i0p = sum(i0jp,1);
    phise0 = 2/f*asinh(if0/2/i0n) + Ussn0 + Rfn*iapp;
    phise3 = 2/f*asinh(if3/2/i0p) + Ussp3 - Rfp*iapp;
    
    % cell voltage
    vcell(index) = phise3 + phie3_tilde - phise0;
  end
end % runPSS()

% STUFF Stuff a structure of optimization parameters into a vector.
function vector = stuff(structure,pinned)
  paramnames = {'psi', 'QAh'};
  nparam = length(paramnames);
  npin = length(fieldnames(pinned));
  vector = zeros(nparam-npin,1);
  cursor = 1;
  for k = 1:nparam
    pname = paramnames{k};
    if ~isfield(pinned,pname)
      if any(strcmp(pname,{'DsrefNEG','DsrefPOS'}))
        % Use log scale for diffusion coefficients.
        vector(cursor) = log10(structure.(pname));
      else
        vector(cursor) = structure.(pname);
      end
      cursor = cursor + 1;
    end
  end % for
end % stuff()

% EXPLODE Unstuff a vector of optimization parameters into a structure.
function structure = explode(vector,pinned)
  paramnames = {'psi', 'QAh'};
  nparam = length(paramnames);
  structure = struct;
  cursor = 1;
  for k = 1:nparam
    pname = paramnames{k};
    if isfield(pinned,pname)
      structure.(pname) = pinned.(pname);
    else
      if any(strcmp(pname,{'DsrefNEG','DsrefPOS'}))
        % Use log scale for diffusion coefficients.
        structure.(pname) = 10^(vector(cursor));
      else
        structure.(pname) = vector(cursor);
      end
      cursor = cursor + 1;
    end
  end % for
end % explode()

% EVALSTRINGS Evalulate any strings in a nested struct using eval().
% !!! DO NOT REPLACE `params` WITH `~`, as call to eval() may use it!
function s = evalstrings(s,params)  %#ok<INUSD> 
  secnames = fieldnames(s);
  for ks = 1:length(secnames)
    sname = secnames{ks};
    sec = s.(sname);
    if isstruct(sec)
      paramnames = fieldnames(sec);
      for kp = 1:length(paramnames)
        pname = paramnames{kp};
        value = sec.(pname);
        if ischar(value) || isstring(value)
          s.(sname).(pname) = eval(value);
        end
      end % for
    else
      value = sec;
      if ischar(value) || isstring(value)
        s.(sname) = eval(value);
      end
    end
  end % for
end % function