%FITOCP Regress operating composition boundaries to OCV data.
%
% -- Usage --
% cellData = fitOCV(cellspec,config)
% cellData = fitOCV(...,'OCPEstimators',OCPEst)
% cellData = fitOCV(...,'DAType',DAType)
% cellData = fitOCV(...,'OCPResolution',OCPRes)
% cellData = fitOCV(...,'RegressionResolution',RegRes)
%
% -- Input --
% cellspec    = cell specification generated by labcell() function
% config      = struct of configuration constants
% OCPEst      = cell array of the OCP estimators to use, choose from:
%                 'resistanceBlend'
%                 'voltageAverage'
%                 'socAverage'
%                 'diagonalAverage' (see additional option below)
%               (default diagonalAverage)
%               **only the parameters from the first specified estimator
%               are saved to the output cellData struct**
% DAType     = default type of diagonal averaging to perform:
%                 'useChg': use shifted charge curve
%                 'useDis': use shifted discharge curve
%                 'useAvg': average the shifted dis/charge curves
%               (default 'useAvg')
% OCPRes      = number of points to use to approximate dis/charge curves
%               (default 1000)
% RegRes      = number of composition points to use in model regression
%               (default 500)
% 
% The `config` struct has the following fields. Fields
% suffixed by an asterisk (*) are optional.
%
%   .vmin     = minimum operating voltage [V]
%   .vmax     = maximum operating voltage [V]
%   .init     = struct of initial values of MSMR params (see below)
%   .lb       = struct of lower bounds of MSMR params (see below)
%   .ub       = struct of upper bounds of MSMR params (see below)
%   .optw     = weight to assign to differential capacity in cost function
%   .datype*  = type of diagonal averaging ('useChg','useDis', or 'useAvg')
%   .daxcorrfiltz* = function used to limit SOC interval over which xcorr 
%                    of dis/charge dZ-vs-Z curves are evalulated (see below)
%   .daxcorrfiltv* = function used to limit voltage interval over which xcorr 
%                    of dis/charge dZ-vs-V curves are evalulated (see below)
%
%   The .daxcorrfiltz and .daxcorrfiltv functions accept vectors
%   of the electrode SOC or voltage, respectively, and return a vector of
%   logical indicies describing where the cross-correlation should be
%   computed. To use multiple distinct intervals and average the xcorr over
%   the intervals, return a cell array with an entry for each interval.
%
% -- Output --
% cellData
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function fitData = fitOCV(cellspec,config,varargin)
  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addRequired('config',@(x)isscalar(x)&&isstruct(x));
  parser.addParameter('DAType','useAvg',@(x)any(strcmp(x,{'useChg','useDis','useAvg'})));
  parser.addParameter('DAxcorrfiltz',@(z)true(size(z)));
  parser.addParameter('DAxcorrfiltv',@(z)true(size(z)));
  parser.addParameter('OCPEstimators',{'diagonalAverage'},@iscellstr);
  parser.addParameter('OCPResolution',1000,@(x)isscalar(x)&&x>0);
  parser.addParameter('RegressionResolution',500,@(x)isscalar(x)&&x>0);
  parser.parse(cellspec,config,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  % Fill optional electrode config args
  if ~isfield(config,'datype'),       config.datype = arg.DAType; end
  if ~isfield(config,'daxcorrfiltz'), config.daxcorrfiltz = arg.DAxcorrfiltz; end
  if ~isfield(config,'daxcorrfiltv'), config.daxcorrfiltv = arg.DAxcorrfiltv; end
  
  % Collect arguments.
  cellname = arg.cellspec.name;
  processfile = arg.cellspec.ocv.processfile;
  fitfile = arg.cellspec.ocv.fitfile;
  fitfileOCP = arg.cellspec.ocp.fitfile;
  timestamp = arg.cellspec.timestamp__;
  
  outp.print('Started fitOCV %s\n',cellname);
  
  % Load processed cell data.
  processData = load(processfile);
  cellData = processData.cellData;
  ocpdat = load(fitfileOCP);
  
  % Process negative electrode
  ocvrel = estimateOCV(processData,config,arg.OCPResolution,arg.OCPEstimators);
  clear fit;
  for kt = length(ocvrel):-1:1  % iterate temperatures
    ocvdat = ocvrel(kt);
    TdegC = ocvdat.TdegC;
  
    % Get OCP curves for either electrodes at the corresponding temperature.
    ind = find(ocpdat.cellData.const.TdegC==TdegC,1);
    if isempty(ind)
        error(['Could not match cell OCV @ %.2fdegC to electrode OCP at ' ...
            'same temperature. Cannot continue.'],TdegC);
    end
    ocpfitNEG = ocpdat.ocpfitNEG(ind);
    ocpfitPOS = ocpdat.ocpfitPOS(ind);
    ocprelNEG = ocpdat.ocprelNEG(ind);
    ocprelPOS = ocpdat.ocprelPOS(ind);
  
    fitdat = struct;
    for ke = 1:length(ocvdat.estimates)  % iterate estimators
      ocvest = ocvdat.estimates(ke);
      estname = ocvest.name;
  
      % Locate corresponding estimator for neg/pos electrodes.
      if ~isfield(ocpfitNEG,estname)
          error(['Could not match cell OCV:%s to negative electrode ' ...
              'OCP with same estimator. Cannot continue.'],estname);
      end
      if ~isfield(ocpfitPOS,estname)
          error(['Could not match cell OCV:%s to positive electrode ' ...
              'OCP with same estimator. Cannot continue.'],estname);
      end
      ocpestNEG = ocpfitNEG.(estname);
      ocpestPOS = ocpfitPOS.(estname);
      for kr = 1:length(ocprelNEG.estimates)
        if strcmp(estname,ocprelNEG.estimates(kr).name)
          ocprelNEG0 = ocprelNEG.estimates(kr);
          break;
        end
      end
      for kr = 1:length(ocprelPOS.estimates)
        if strcmp(estname,ocprelPOS.estimates(kr).name)
          ocprelPOS0 = ocprelPOS.estimates(kr);
          break;
        end
      end
  
      % Perform model regression.
      tmp = fitCompositionBounds(ocvest,ocpestNEG,ocpestPOS,config,arg);
      tmp.estname = estname;
      tmp.ocpdatNEG = ocpestNEG;
      tmp.ocpdatPOS = ocpestPOS;
      tmp.ocprelNEG = ocprelNEG0;
      tmp.ocprelPOS = ocprelPOS0;
      fitdat.(estname) = tmp;
    end
    fit(kt) = fitdat;
  end
  
  % Save MSMR params to cellData struct.
  % (use values from first OCV estimator.)
  tmp = [fit.(arg.OCPEstimators{1})];
  params = [tmp.params];
  ocpfitNEG = [tmp.ocpdatNEG];
  paramsNEG = [ocpfitNEG.params];
  ocprelNEG = [tmp.ocprelNEG];
  ocpfitPOS = [tmp.ocpdatPOS];
  paramsPOS = [ocpfitPOS.params];
  ocprelPOS = [tmp.ocprelPOS];
  cellData.neg.U0 = [paramsNEG.U0];
  cellData.neg.X = [paramsNEG.X];
  cellData.neg.omega = [paramsNEG.omega];
  cellData.neg.theta0 = [params.theta0n];
  cellData.neg.theta100 = [params.theta100n];
  cellData.pos.U0 = [paramsPOS.U0];
  cellData.pos.X = [paramsPOS.X];
  cellData.pos.omega = [paramsPOS.omega];
  cellData.pos.theta0 = [params.theta0p];
  cellData.pos.theta100 = [params.theta100p];
  
  % Collect and save output data.
  fitData = struct;
  fitData.cellData = cellData;
  fitData.ocvrel = ocvrel;
  fitData.ocvfit = fit;
  fitData.ocprelNEG = ocprelNEG;
  fitData.ocprelPOS = ocprelPOS;
  fitData.origin__ = 'fitOCV';
  fitData.arg__ = arg;
  fitData.timestamp__ = timestamp;
  save(fitfile,'-struct','fitData');
  
  outp.print('Finished fitOCV %s\n\n',cellname);
end


% Utility functions -------------------------------------------------------

%FITMSMR Regress composition bounds to an OCV estimate.
%
% -- Usage --
% data = FITCOMPOSITIONBOUNDS(ocvest,config,arg)
%
% -- Input --
% ocvest    = single OCV estimate (from estimateOCV fn)
% ocpestNEG = single OCP estimate for neg electrode
% ocpestPOS = single OCP estimate for pos electrode
% config = struct of configuration data
% arg    = struct of arguments passed to top-level fitOCV fn
%
% -- Output --
% The output, data, is a struct with these fields:
%   .params  : struct of composition bounds 
%              (theta0n, theta100n, theta0p, theta100p)
%   .TdegC   : temperature [degC]

function data = fitCompositionBounds(ocvest, ocpestNEG, ocpestPOS, config, arg)

  persistent fig;
  if outp.plot && (isempty(fig) || ~ishandle(fig))
    fig = figure('Name','Composition Bounds Regression', ...
        'WindowStyle','docked');
  end

  % Constants.
  vmin = config.vmin;         % min electrode voltage [V]
  vmax = config.vmax;         % max electrode voltage [V]
  w = config.optw;            % Relative weight to assign to dZdV residuals
  TdegC = ocvest.TdegC;       % Temperature [degC]
  estname = ocvest.name;      % name of OCP estimator
  npoints = arg.RegressionResolution;
  
  % Collect lab OCV curve. 
  % Force SOC vector to span entire 0% to 100% range.
  Zcell = ocvest.Z;
  Zcell = Zcell - min(Zcell);
  Zcell = Zcell/max(Zcell);
  Ucell = ocvest.U;
  
  % Collect OCP curve for negative electrode.
  ocpNEG = MSMR(ocpestNEG.params).ocp( ...
      'thetamin',0.001,'thetamax',0.999, ...
      'npoints',npoints,'TdegC',ocpestNEG.TdegC);
  Zneg = ocpNEG.theta;
  Uneg = ocpNEG.Uocp;
  
  % Collect OCP curve for positive electrode.
  ocpPOS = MSMR(ocpestPOS.params).ocp( ...
      'thetamin',0.001,'thetamax',0.999, ...
      'npoints',npoints,'TdegC',ocpestPOS.TdegC);
  Zpos = ocpPOS.theta;
  Upos = ocpPOS.Uocp;
  
  % Define common SOC vector for interpolation.
  Zcommon = linspace(0,1,npoints);
  
  % Setup live plot.
  if outp.plot
    % For some reason, we have to do this twice, or the docked figure
    % is cut off.
    for k = 1:2
      figure(fig); clf;
      subplot(1,2,1);
      line_OcpLab = plot(NaN,NaN,'k:'); hold on;
      line_OcpMod = plot(NaN,NaN,'m');
      xlabel('Cell SOC, $z$', ...
          'Interpreter','latex');
      ylabel('$U_\mathrm{ocp}$ [$\mathrm{V}$]','Interpreter','latex');
      title('Full-Cell Composition-Bounds Regression');
      legend('Lab','Model','Location','southeast');
      xlim([0 1]);
      ylim([vmin vmax]);
      subplot(1,2,2);
      line_dOcpLab = plot(NaN,NaN,'k:'); hold on;
      line_dOcpMod = plot(NaN,NaN,'m');
      xlabel(['Absolute Differential Capacity, ' ...
          '$\mathrm{d}\tilde{\theta}_\mathrm{s}/\mathrm{d}U$ ' ...
          '[$\mathrm{V}^{-1}$]'], ...
          'Interpreter','latex');
      ylabel('$U_\mathrm{ocp}$ [$\mathrm{V}$]','Interpreter','latex');
      title(sprintf('%s / %.2f\\circC',estname,TdegC));
      ylim([vmin vmax]);
      thesisFormat;
      drawnow;
    end
  end
  
  % Reduce initial struct to vector for optimization routine.
  init = pack(config.init);
  
  % Bounds.
  lb = zeros(4,1);  % minimum composition is 0
  ub = ones(4,1);   % maximum composition is 1
  
  % These matricies implement the following constraints:
  % 1. theta0n   < theta100n  -->  theta0n - theta100n < 0
  % 2. theta100p < theta0p    --> -theta0p + theta100p < 0
  A = [1 -1 0 0; 0 0 -1 1];
  B = [0; 0];
  
  % Track best RMSE between model and lab curves throughout optimization.
  bestRMSE = Inf;
  
  % Configure and run optimization.
  outp.print('Regression: %s %.2fdegC\n',estname,TdegC);
  optsFmincon = optimoptions(@fmincon, ...
      'Display','off',...
      'MaxFunEvals',20000,... % default: 3000
      'MaxIter',10000,...     % default: 1000
      'TolFun',1e-10,...      % default: 1e-6
      'TolX',1e-15,...        % default: 1e-10
      'TolCon',1e-20...       % default: 1e-6
  );
  vect = fmincon(@cost,init,A,B,[],[],lb,ub,[],optsFmincon);
  paramsfin = unpack(vect);
  outp.info('  theta0n       : ');
  outp.info('%5.3f ',paramsfin.theta0n);
  outp.ln;
  outp.info('  theta100n     : ');
  outp.info('%5.3f ',paramsfin.theta100n);
  outp.ln;
  outp.info('  theta0p       : ');
  outp.info('%5.3f ',paramsfin.theta0p);
  outp.ln;
  outp.info('  theta100p     : ');
  outp.info('%5.3f ',paramsfin.theta100p);
  outp.ln;
  
  % Collect output.
  data.params = paramsfin;
  data.TdegC = TdegC;
  
    function [Jcost, rmse] = cost(vect)
      % Unpack parameter vector
      params = unpack(vect);
  
      % Get cell OCV and differential capacity predicted by the model.
      ZmodNEG = linspace(params.theta0n,params.theta100n,npoints);
      ZmodPOS = linspace(params.theta0p,params.theta100p,npoints);
      UmodNEG = linearinterp(Zneg,Uneg,ZmodNEG);
      UmodPOS = linearinterp(Zpos,Upos,ZmodPOS);
      UmodFULL = UmodPOS - UmodNEG;  % OCV of full cell
      dZmodFULL = roughdiff(Zcommon,UmodFULL);
      
      % Get true OCV and differential capacity.
      UlabFULL = linearinterp(Zcell,Ucell,Zcommon);
      dZlabFULL = roughdiff(Zcommon,UlabFULL);
  
      % Compute residuals.
      uResid = UmodFULL(:) - UlabFULL(:);
      dzResid = dZmodFULL(:) - dZlabFULL(:);
  
      % Compute cost and root-mean-square error (RMSE).
      Jcost = sum(uResid.^2) + w*sum(dzResid.^2);
      rmse  = rms(uResid);
  
      if rmse < bestRMSE
        if ~isinf(bestRMSE)
          outp.info(repmat('\b',1,length('  RMSE  : xxxxxxxxx mV')));
        end
        outp.info('  RMSE  : %8.3f mV\n',1000*rmse);
        bestRMSE = rmse;
        
        if outp.plot && ishandle(fig)
          set(line_OcpLab,'XData',Zcommon,'Ydata',UlabFULL);
          set(line_dOcpLab,'XData',dZlabFULL,'Ydata',UlabFULL);
          set(line_OcpMod,'XData',Zcommon,'Ydata',UmodFULL);
          set(line_dOcpMod,'XData',dZmodFULL,'Ydata',UmodFULL);
          drawnow;
        end
      end
    end
    
    % PACK Reduce parameter structure to a vector of parameters.
    function vect = pack(params)
      vect = [params.theta0n; params.theta100n; params.theta0p; params.theta100p];
    end
    
    % UNPACK Restore a MSMR parameter structure from the given vector.
    function params = unpack(vect)
      vect = vect(:);
      params.theta0n = vect(1);
      params.theta100n = vect(2);
      params.theta0p = vect(3);
      params.theta100p = vect(4);
    end
  end