% FITOCP Regress processed OCP data to the MSMR model.
%
% -- Usage --
% cellData = fitOCP(cellspec,configNEG,configPOS)
% cellData = fitOCP(...,'OCPEstimators',OCPEst)
% cellData = fitOCP(...,'DAType',DAType)
% cellData = fitOCP(...,'OCPResolution',OCPRes)
%
% -- Input --
% cellspec    = cell specification generated by labcell() function
% configNEG,  = structs of configuration constants for the negative
% configPOS     and positive electrodes (see below)
% OCPEst      = cell array of the OCP estimators to use, choose from:
%                 'resistanceBlend'
%                 'voltageAverage'
%                 'socAverage'
%                 'diagonalAverage' (see additional option below)
%               (default diagonalAverage)
%               **only the parameters from the first specified estimator
%               are saved to the output cellData struct**
% DAType     = default type of diagonal averaging to perform:
%                 'useChg': use shifted charge curve
%                 'useDis': use shifted discharge curve
%                 'useAvg': average the shifted dis/charge curves
%               (default 'useAvg')
% OCPRes      = number of points to use to approximate dis/charge curves
%               (default 1000)
% 
% The configNEG and configPOS structs have the following fields. Fields
% suffixed by an asterisk (*) are optional.
%
%   .vmin     = minimum operating voltage [V]
%   .vmax     = maximum operating voltage [V]
%   .init     = struct of initial values of MSMR params (see below)
%   .lb       = struct of lower bounds of MSMR params (see below)
%   .ub       = struct of upper bounds of MSMR params (see below)
%   .optw     = weight to assign to differential capacity in cost function
%   .datype*  = type of diagonal averaging ('useChg','useDis', or 'useAvg')
%   .daxcorrfiltz* = function used to limit SOC interval over which xcorr 
%                    of dis/charge dZ-vs-Z curves are evalulated (see below)
%   .daxcorrfiltv* = function used to limit voltage interval over which xcorr 
%                    of dis/charge dZ-vs-V curves are evalulated (see below)
%
%   The .init, .lb, and .ub structs have the following fields:
%
%     .U0       = MSMR { U0_j    } params in column vector [V]
%     .X        = MSMR { X_j     } params in column vector [-]
%     .omega    = MSMR { omega_j } params in column vector [-]
%     .thetamin = minumum electrode composition [-]
%     .thetamax = maximum electrode composition [-]
%
%   The .daxcorrfiltz and .daxcorrfiltv functions accept vectors
%   of the electrode SOC or voltage, respectively, and return a vector of
%   logical indicies describing where the cross-correlation should be
%   computed. To use multiple distinct intervals and average the xcorr over
%   the intervals, return a cell array with an entry for each interval.
%
% -- Output --
%
%
% -- Known issues --
% * MSMR regression is not the most robust
%   TODO: use particle-swarm opt with fmincon to increase robustness
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function fitData = fitOCP(cellspec,configNEG,configPOS,varargin)

  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addRequired('configNEG',@(x)isscalar(x)&&isstruct(x));
  parser.addRequired('configPOS',@(x)isscalar(x)&&isstruct(x));
  parser.addParameter('DAType','useAvg',@(x)any(strcmp(x,{'useChg','useDis','useAvg'})));
  parser.addParameter('DAxcorrfiltz',@(z)true(size(z)));
  parser.addParameter('DAxcorrfiltv',@(z)true(size(z)));
  parser.addParameter( ...
      'OCPEstimators', ...
      {'diagonalAverage'}, ...
      @iscellstr);
  parser.addParameter('OCPResolution',1000,@(x)isscalar(x)&&x>0);
  parser.parse(cellspec,configNEG,configPOS,varargin{:});
  arg = parser.Results;  % struct of validated arguments
  
  % Fill optional electrode config args
  if ~isfield(configNEG,'datype'),       configNEG.datype = arg.DAType; end
  if ~isfield(configNEG,'daxcorrfiltz'), configNEG.daxcorrfiltz = arg.DAxcorrfiltz; end
  if ~isfield(configNEG,'daxcorrfiltv'), configNEG.daxcorrfiltv = arg.DAxcorrfiltv; end
  if ~isfield(configPOS,'datype'),       configPOS.datype = arg.DAType; end
  if ~isfield(configPOS,'daxcorrfiltz'), configPOS.daxcorrfiltz = arg.DAxcorrfiltz; end
  if ~isfield(configPOS,'daxcorrfiltv'), configPOS.daxcorrfiltv = arg.DAxcorrfiltv; end
  
  % Collect arguments.
  cellname = arg.cellspec.name;
  processfile = arg.cellspec.ocp.processfile;
  fitfile = arg.cellspec.ocp.fitfile;
  hcname = arg.cellspec.hcname;
  timestamp = arg.cellspec.timestamp__;
  
  outp.print('Started fitOCP %s%s\n',cellname,hcname);
  % Warn about known issues.
  outp.warning(['Known issue of fitOCP: Lower/upper bounds on X(end) have ' ...
      'no effect due to a limitation design of the MSMR regression routine.\n']);
  
  % Load processed cell data.
  processData = load(processfile);
  cellData = processData.cellData;
  
  % Process negative electrode
  ocprelNEG = estimateOCV(processData.neg,configNEG,arg.OCPResolution,arg.OCPEstimators);
  clear fitNEG;
  for kt = length(ocprelNEG):-1:1  % iterate temperatures
    ocpdat = ocprelNEG(kt);
    fitdat = struct;
    for ke = 1:length(ocpdat.estimates)  % iterate estimators
      ocpest = ocpdat.estimates(ke);
      fit = fitMSMR(ocpest,configNEG,arg);
      fit.estname = ocpest.name;
      fitdat.(ocpest.name) = fit;
    end
    fitNEG(kt) = fitdat;
  end
  
  % Process positive electrode
  ocprelPOS = estimateOCV(processData.pos,configPOS,arg.OCPResolution,arg.OCPEstimators);
  clear fitPOS;
  for kt = length(ocprelPOS):-1:1  % iterate temperatures
    ocpdat = ocprelPOS(kt);
    fitdat = struct;
    for ke = 1:length(ocpdat.estimates)  % iterate estimators
      ocpest = ocpdat.estimates(ke);
      fit = fitMSMR(ocpest,configPOS,arg);
      fit.estname = ocpest.name;
      fitdat.(ocpest.name) = fit;
    end
    fitPOS(kt) = fitdat;
  end
  
  % Save MSMR params to cellData struct.
  % (use values from first OCP estimator.)
  tmpNEG = [fitNEG.(arg.OCPEstimators{1})];
  tmpPOS = [fitPOS.(arg.OCPEstimators{1})];
  paramsNEG = [tmpNEG.params];
  paramsPOS = [tmpPOS.params];
  pnames = fieldnames(paramsNEG);
  for kp = 1:length(pnames)
    pname = pnames{kp};
    cellData.neg.(pname) = [paramsNEG.(pname)];
    cellData.pos.(pname) = [paramsPOS.(pname)];
  end
  
  % Collect and save output data.
  fitData = struct;
  fitData.cellData = cellData;
  fitData.ocprelNEG = ocprelNEG;
  fitData.ocpfitNEG = fitNEG;
  fitData.ocprelPOS = ocprelPOS;
  fitData.ocpfitPOS = fitPOS;
  fitData.origin__ = 'fitOCP';
  fitData.arg__ = arg;
  fitData.timestamp__ = timestamp;
  save(fitfile,'-struct','fitData');
  
  outp.print('Finished fitOCP %s%s\n\n',cellname,hcname);

end