% FITMSMR Regress the MSMR model to an OCP estimate.
%
% -- Usage --
% data = FITMSMR(ocpest,config)
%
% -- Input --
% ocpest = struct of single OCP estimate (from estimateOCP fn)
% config = struct of electrode configuration data
%
% -- Output --
% The output, data, is a struct with these fields:
%   .params  : struct of MSMR parameters (U, X, omega, thetamin, thetamax)
%   .Zmod    : final model relative composition vector
%   .Umod    : final model OCP vector [V]
%   .dZmod   : final model differentialc capacity vector [1/V]
%   .TdegC   : temperature [degC]
%   .elecname: electrode/cell name (e.g., 'NEG' or 'POS' or 'FULL')
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function data = fitMSMR(ocpest, config, ~)
  
  persistent fig;
  if outp.plot && (isempty(fig) || ~ishandle(fig))
      fig = figure('Name','MSMR Model Regression','WindowStyle','docked');
  end
  
  % Constants.
  J = length(config.init.U0); % number of MSMR galleries
  indU = 1:J;                 % indicies to U0    in param vectors
  indX = J+1:2*J-1;           % indicies to X     in param vectors
  indW = 2*J:3*J-1;           % indicies to omega in param vectors
  w = config.optw;            % Relative weight to assign to dZdV residuals
  TdegC = ocpest.TdegC;       % Temperature [degC]
  elecname = ocpest.elecname; % electrode name ('NEG' or 'POS')
  estname = ocpest.name;      % name of OCP estimator
  vmin = config.vmin;         % min electrode voltage [V]
  vmax = config.vmax;         % max electrode voltage [V]
  Ulab = ocpest.U;            % OCP vector derived from lab data [V]
  Zlab = ocpest.Z;            % SOC vector derived from lab data [-]
  dZlab = abs(ocpest.dZ);     % absolute differential capacity
  Umod = Ulab;                % OCP vector over which to evalulate MSMR model [V]
  diffzlab = max(Zlab) - min(Zlab);
  
  % Setup live plot.
  if outp.plot
    % For some reason, we have to do this twice, or the docked figure
    % is cut off.
    for k = 1:2
      figure(fig); clf;
      subplot(1,2,1);
      line_OcpLab = plot(NaN,NaN,'k:'); hold on;
      line_OcpMod = plot(NaN,NaN,'m');
      xlabel('Relative Composition, $\tilde{\theta}_\mathrm{s}$', ...
          'Interpreter','latex');
      ylabel('$U_\mathrm{ocp}$ [$\mathrm{V}$]','Interpreter','latex');
      title(sprintf('%s Electrode MSMR Regression',ocpest.elecname));
      legend('Lab','Model');
      xlim([0 1]);
      ylim([vmin vmax]);
      subplot(1,2,2);
      line_dOcpLab = plot(NaN,NaN,'k:'); hold on;
      line_dOcpMod = plot(NaN,NaN,'m');
      xlabel(['Absolute Differential Capacity, ' ...
          '$\mathrm{d}\tilde{\theta}_\mathrm{s}/\mathrm{d}U$ ' ...
          '[$\mathrm{V}^{-1}$]'], ...
          'Interpreter','latex');
      ylabel('$U_\mathrm{ocp}$ [$\mathrm{V}$]','Interpreter','latex');
      title(sprintf('%s / %.2f\\circC',estname,TdegC));
      ylim([vmin vmax]);
      thesisFormat;
      drawnow;
      pause(0.5);
    end
  end
  
  % Reduce lower/upper bounds and initial structs to vectors for optimization
  % routine.
  lb = stuff(config.lb);
  ub = stuff(config.ub);
  init = stuff(config.init);
  
  % Track best RMSE between model and lab curves throughout optimization.
  bestRMSE = Inf;
  
  % Configure and run optimization.
  outp.print('MSMR Regression: %s %s %.2fdegC\n',elecname,estname,TdegC);
  opts = optimoptions(@fmincon, ...
      'Display','off',...
      'MaxFunEvals',20000,... % default: 3000
      'MaxIter',10000,...     % default: 1000
      'TolFun',1e-10,...      % default: 1e-6
      'TolX',1e-15,...        % default: 1e-10
      'TolCon',1e-20...       % default: 1e-6
  );
  vect = fmincon(@cost,init,[],[],[],[],lb,ub,[],opts);
  paramsfin = explode(vect);
  [Zmodfin,dZmodfin] = compositionMSMR(Umod,paramsfin,TdegC);
  outp.info('  U0       : ');
  outp.info('%5.3f ',paramsfin.U0);
  outp.ln;
  outp.info('  X        : ');
  outp.info('%5.3f ',paramsfin.X);
  outp.ln;
  outp.info('  omega    : ');
  outp.info('%5.3f ',paramsfin.omega);
  outp.ln;
  outp.info('  thetamin : ');
  outp.info('%5.3f ',paramsfin.thetamin);
  outp.ln;
  outp.info('  thetamax : ');
  outp.info('%5.3f ',paramsfin.thetamax);
  outp.ln;
  
  % Collect output.
  data.params = paramsfin;
  data.Zmod = Zmodfin;
  data.Umod = Umod;
  data.dZmod = dZmodfin;
  data.TdegC = TdegC;
  data.elecname = elecname;
  
  
  function [Jcost, rmse] = cost(vect)
    % Unpack parameter vector
    params = explode(vect);
    
    % Evalulate MSMR model over same Uocp vector as lab data.
    [Zmod,dZmod] = compositionMSMR(Umod,params,TdegC); 

    % Convert model vectors over absolute composition to relative
    % composition for comparison with data.
    Zmod = (Zmod - params.thetamin)/(params.thetamax - params.thetamin);
    Zmod = min(Zlab) + Zmod*diffzlab;
    dZmod = dZmod*diffzlab/(params.thetamax - params.thetamin);

    % Compute residuals.
    zResid = Zmod(:) - Zlab(:);
    dzResid = dZmod(:) - dZlab(:);

    % Compute cost and root-mean-square error (RMSE).
    Jcost = sum(zResid.^2) + w*sum(dzResid.^2);
    rmse  = rms(zResid);

    if rmse < bestRMSE
      if ~isinf(bestRMSE)
        outp.info(repmat('\b',1,length('  RMSE  : xxxxxxxxx')));
      end
      outp.info('  RMSE  : %8.3f\n',1000*rmse);
      bestRMSE = rmse;
      
      if outp.plot && ishandle(fig)
        set(line_OcpLab,'XData',Zlab,'Ydata',Ulab);
        set(line_dOcpLab,'XData',dZlab,'Ydata',Ulab);
        set(line_OcpMod,'XData',Zmod,'Ydata',Umod);
        set(line_dOcpMod,'XData',dZmod,'Ydata',Umod);
        drawnow;
      end
    end
  end
  
  % STUFF Reduce an MSMR parameter structure to a vector of parameters.
  function vect = stuff(params)
    X = params.X(1:end-1); % omit last gallery, as we'll compute X_end=1-sum(X_other)
    vect = [params.U0(:); X(:); params.omega(:); params.thetamin; params.thetamax];
  end
  
  % EXPLODE Restore a MSMR parameter structure from the given vector.
  function params = explode(vect)
    vect = vect(:);
    params.U0 = vect(indU);  
    params.X = [vect(indX); 1-sum(vect(indX))]; 
    params.omega = vect(indW);
    params.thetamin = vect(end-1); 
    params.thetamax = vect(end); 
  end  
end
  
%COMPSITIONMSMR Calculate composition predicted by MSMR model over OCP.
function [Z, dZ] = compositionMSMR(Uvect,params,TdegC)
  
  % Constants.
  R = 8.3144598;  % Molar gas constant [J/mol K]
  F = 96485.3329; % Faraday constant [C/mol]
  
  U0 = params.U0(:);
  X = params.X(:);
  omega = params.omega(:);
  f = F/R/(TdegC+273.15);
  Uvect = Uvect(:).';
  gj = exp(f*(Uvect-U0)./omega);
  xj = X./(1+gj);
  Z = sum(xj,1);
  dZj = (X./omega).*gj./(1+gj).^2;
  dZj(isnan(dZj)) = 0; % !!! when gj->Inf, (gj./(1+gj).^2)->0
  dZ = f*sum(dZj,1);
  Z = Z(:);
  dZ = dZ(:);
end