% FITEIS Regress transfer-function (TF) model to linear EIS measurements.
%
% Estimates all remaining parameters of the PDE model except for psi, which
% may take any positive value. The ratios kD/psi, qen/psi, qes/psi, and
% qep/psi are the identifiable quantities. Run fitPSS() after this function
% to estimate the value of psi and fix the values of kD, qen, qes, and qep.
%
% Optimizer: particleswarm and fmincon. Best results with the graphical
% user interface. Try tuning the parameter values manually in addition to
% running PSO and fmincon from the GUI.
%
% Before running the TF model regression, you verify the integrity
% of EIS measurements using the linKK() function.
%
% NOTE 2: Currently only implements "strategy 1" for regressing the TF
%         model. Assumes Baker-Verbrugge solid-diffusivity and MSMR 
%         kinetics. Arrhenius equations for temperature variation.
%         **Improved regression accuracy possible by fitting individual 
%         diffusivity and exchange current values at each SOC setpoint
%         ("strategy 2").**
%
% -- Usage --
% outData = FITEIS(cellspec,model,Tstring)
% outData = FITEIS(cellspec,model,Tstring,ui)
% outData = FITEIS(...,'Np',NP)
% outData = FITEIS(...,'filterFcn',FILTFN)
%
% -- Input --
% cellspec    = cell specification generated by the labcell() function
% model       = optutil model for regression (from optutil.modelspec)
% Tstring     = string selecting the temperature(s) at which to perform the
%               model regression. If two or more temperatures are given,
%               Arrhenius equations are used to describe varation of the
%               parameter values with temperature. Example: '0degC 25degC'.
% ui          = 'cli' for command-line interface or 'gui' for graphical 
%               user interface (DEFAULT 'gui').
% NP          = number of particles to use in particle swarm optimization
%               (CLI only, DEFAULT 100).
% FILTFN      = handle to function that accepts EIS data structure and
%               returns a modified version of the EIS data structure. Use
%               to exclude SOC setpoints and/or frequency points.
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function outData = fitEIS(cellspec,model,varargin)
  
  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addRequired('model',@(x)isscalar(x)&&isstruct(x));
  parser.addRequired('Tstring',@(x)ischar(x));
  parser.addOptional('ui','cli',@(x)any(strcmp(x,{'cli','gui'})));
  parser.addParameter('Np',100,@(x)isscalar(x)&&x>10);
  parser.addParameter('filterFcn',@(eis)eis); % default no filtering
  parser.parse(cellspec,model,varargin{:});
  arg = parser.Results; % struct of validated arguments
  
  % Collect cell metadata.
  cellname = arg.cellspec.name;
  processfile = arg.cellspec.eis.processfile;
  fitfile = arg.cellspec.eis.fitfile;
  timestamp = arg.cellspec.timestamp__;
  
  outp.print('Started fitEIS %s\n',cellname);
  
  % Load processed pulse data.
  processData = load(processfile);
  cellData = processData.cellData; % struct of param values previously identified
  eisVect = processData.eisVect; % vector struct of EIS at each temperature
  
  % Get selected temperature(s).
  Tvect = cellfun(@(x)str2double(x(1:end-4)),strsplit(arg.Tstring));
  indT = find(sum([eisVect.TdegC]==Tvect'));
  if isempty(indT)
    error(['Could not find EIS datasets at requested temperatures. ' ...
        'Cannot continue.\n']);
  end
  if length(indT) < 1
    error(['Need datasets at at least one temperature to run EIS ' ...
        'regression. Cannot continue.'])
  end
  
  % Structure to store all arguments passed to cost function.
  p = struct;
  
  % Collect struct of EIS measurements at selected temperatures.
  eisData = eisVect(indT);
  clear tempSet;
  for kt = length(eisData):-1:1
    tmp = struct;
    tmp.TdegC = Tvect(kt);
    tmp.socPct = eisData(kt).socPct;  % vector of SOC setpoints
    tmp.freq = eisData(kt).reg.freq;  % use regression dataset ('reg')
    tmp.Z = eisData(kt).reg.Z;

    % Fetch OCP curves.
    % NEG - use regressed MSMR model (absolute OCP)
    thetaNEG = cellData.neg.theta0 + ...
        (tmp.socPct/100)*(cellData.neg.theta100-cellData.neg.theta0);
    ocpNEG = MSMR(cellData.neg).ocp('theta',thetaNEG,'TdegC',Tvect(kt));
    tmp.thetaNEG = ocpNEG.theta;
    tmp.UocpNEG = ocpNEG.Uocp;
    tmp.dUocpNEG = ocpNEG.dUocp;
    % POS - use regressed MSMR model (absolute OCP)
    thetaPOS = cellData.pos.theta0 + ... 
        (tmp.socPct/100)*(cellData.pos.theta100-cellData.pos.theta0);
    ocpPOS = MSMR(cellData.pos).ocp('theta',thetaPOS,'TdegC',Tvect(kt));
    tmp.thetaPOS = ocpPOS.theta;
    tmp.UocpPOS = ocpPOS.Uocp;
    tmp.dUocpPOS = ocpPOS.dUocp;

    % Apply data filter function.
    tmp = arg.filterFcn(tmp);

    tempSet(kt) = tmp;   
  end
  p.Tvect = Tvect;
  p.ntemp = length(Tvect);
  p.tempSet = tempSet;
  
  if p.ntemp == 1
    % Single temperature - exclude activation energy parameters.
    p.utilopts = {'notemp'};
    p.TrefdegC = Tvect(1);
  else
    p.utilopts = {};
    p.TrefdegC = 25;
  end
  
  % Add fixed parameters to model.
  model = arg.model;
  model.neg.U0 = optutil.param('fix',cellData.neg.U0);
  model.neg.X = optutil.param('fix',cellData.neg.X);
  model.neg.omega = optutil.param('fix',cellData.neg.omega);
  model.neg.theta0 = optutil.param('fix',cellData.neg.theta0);
  model.neg.theta100 = optutil.param('fix',cellData.neg.theta100);
  model.pos.U0 = optutil.param('fix',cellData.pos.U0);
  model.pos.X = optutil.param('fix',cellData.pos.X);
  model.pos.omega = optutil.param('fix',cellData.pos.omega);
  model.pos.theta0 = optutil.param('fix',cellData.pos.theta0);
  model.pos.theta100 = optutil.param('fix',cellData.pos.theta100);
  model.const.Q = optutil.param('fix',cellData.const.Q);
  modelspec = optutil.modelspec(model,'TrefdegC',p.TrefdegC);
  p.modelspec = modelspec;
  
  init = optutil.pack('init',modelspec,[],'coerce',p.utilopts{:});
  lb = optutil.pack('lb',modelspec,[],'coerce',p.utilopts{:});  % vector of lower bounds
  ub = optutil.pack('ub',modelspec,[],'coerce',p.utilopts{:});  % vector of upper bounds
  initStruct = optutil.unpack(init,modelspec,p.utilopts{:});
  lbStruct = optutil.unpack(lb,modelspec,p.utilopts{:});
  ubStruct = optutil.unpack(ub,modelspec,p.utilopts{:});
  
  if strcmp(arg.ui,'gui')
    % Run GUI.
    [plotInit, plotUpdate] = uiEIS(p);
    data = optutil.uiparticleswarm( ...
        @(x)uicost(x,p), ...
        modelspec,initStruct,lbStruct,ubStruct, ...
        'PlotInitializeFcn',plotInit, ...
        'PlotUpdateFcn',plotUpdate, ...
        'UseParallel',false, ...
        'UtilOpts',p.utilopts ...
    );
    paramStruct = data.values;
  else
    % Configure and run optimizer.
    % -- PSO+fmincon --
    Np = arg.Np;       % number of particles
    pop0 = init(:).';  % (partial) initial population
    optionsFMINCON = optimoptions(@fmincon,'Display','off',...
        'MaxFunEvals',1e6,'MaxIter',1e3,...
        'TolFun',1e-20,'TolX',1e-20,'TolCon',1e-20);
    optionsPSO = optimoptions(@particleswarm,...
        'Display','off','UseParallel',false,...
        'FunctionTolerance',1e-20,'SwarmSize',Np,...
        'MaxIterations',100,'MaxStallIterations',20,...
        'InitialSwarmMatrix',pop0,'FunValCheck','off',...
        'HybridFcn',{@fmincon,optionsFMINCON});
    outp.info('Running TF regression...\n');
    paramVect = particleswarm(@(x)cost(x,p),length(init),lb,ub,optionsPSO);
    paramStruct = optutil.unpack(paramVect,modelspec,p.utilopts{:});
  end % else
  
  % Save cell parameter values (these apply at TrefdegC).
  secnames = fieldnames(paramStruct);
  for ks = 1:length(secnames)
    secname = secnames{ks};
    sec = paramStruct.(secname);
    paramnames = fieldnames(sec);
    for kp = 1:length(paramnames)
      paramname = paramnames{kp};
      val = sec.(paramname);
      cellData.(secname).(paramname) = val;
    end % for
  end % for
  cellData.const.TdegC = p.modelspec.Tref-273.15;
  
  if p.ntemp == 1
    % Single temperature - remove Eact parameter values.
    secnames = fieldnames(cellData);
    for ks = 1:length(secnames)
      secname = secnames{ks};
      sec = paramStruct.(secname);
      paramnames = fieldnames(sec);
      for kp = 1:length(paramnames)
        paramname = paramnames{kp};
        if endsWith(paramname,'_Eact')
          cellData.(secname) = rmfield(cellData.(secname),paramname);
        end
      end % for
    end % for
  end % if
  
  % Collect and save output data.
  outData = struct;
  outData.cellData = cellData;
  outData.modelspec = p.modelspec;
  outData.origin__ = 'fitEIS';
  outData.arg__ = arg;
  outData.timestamp__ = timestamp;
  save(fitfile,'-struct','outData');
  
  outp.print('Finished fitEIS %s\n',cellname);

end


% Utility functions -------------------------------------------------------

% COST Cost function for the model regression.
function J = cost(vect,param)

  Tvect = param.Tvect;
  ntemp = length(param.Tvect);
  
  ZLAB = cell(ntemp,1);
  for kt = 1:length(param.Tvect)
      ZLAB{kt} = param.tempSet(kt).Z;
  end
  
  % Persistent variables (for plots and debug output).
  persistent minJ bestModel bscount bscount2 evalnum fig lines_ZMOD;
  if isempty(minJ)
    minJ = Inf;
  end
  if isempty(bestModel)
    bestModel = struct;
  end
  if isempty(bscount)
    bscount = 0;
  end
  if isempty(bscount2)
    bscount2 = 0;
  end
  if isempty(evalnum)
      evalnum = 0;
  end
  if outp.plot && (isempty(fig) || ~ishandle(fig))
    minJ = Inf;
    bestModel = struct;
    bscount = 0;
    bscount2 = 0;
    evalnum = 0;
    outp.charcount(0);
    fig = figure( ...
        'Name','EIS Regression', ...
        'WindowStyle','normal');
    lines_ZLAB = cell(ntemp,1);
    lines_ZMOD = cell(ntemp,1);
    for kt = 1:ntemp
      socPct = param.tempSet(kt).socPct;
      colors = cool(length(socPct));
      lines_ZLAB{kt} = gobjects(length(socPct),1);
      lines_ZLAB{kt} = gobjects(length(socPct),1);
      subplot(1,ntemp,kt);
      for kz = 1:length(socPct)
        lines_ZLAB{kt}(kz) = plot( ...
            real(ZLAB{kt}(:,kz)),-imag(ZLAB{kt}(:,kz)), ...
            '.','Color',colors(kz,:)); hold on;
      end % for soc
      for kz = 1:length(socPct)
        lines_ZMOD{kt}(kz) = plot(NaN,NaN,'-','Color',colors(kz,:)); hold on;
      end % for soc
      xlabel('$\mathrm{Re}(Z_\mathrm{cell})$ [$\Omega$]','Interpreter','latex');
      ylabel('$-\mathrm{Im}(Z_\mathrm{cell})$ [$\Omega$]','Interpreter','latex');
      title(sprintf('%.2fdegC',Tvect(kt)));
      legh(1) = plot(NaN,NaN,'k.');
      legh(2) = plot(NaN,NaN,'k-');
      if kt == 1
        legend(legh,'Lab','Model','Location','northwest');
      end
      zlab = ZLAB{kt}(:);
      [~,indrm] = rmoutliers(abs(zlab));
      zlab = zlab(~indrm);
      setAxesNyquist('xdata',real(zlab),'ydata',-imag(zlab));
    end % for
    thesisFormat;
    drawnow;
  end
  
  % Get struct of parameter values.
  model = optutil.unpack(vect,param.modelspec,param.utilopts{:});
  
  % Iterate temperatures, accumulating cost.
  J = 0;
  ZMOD = getImpedance(model,param);
  for kt = 1:length(param.Tvect)
    % Compute the mean square error between the predicted and actual 
    % impedance.
    err = abs(ZLAB{kt}-ZMOD{kt})./abs(ZLAB{kt});
    J = J + mean(err.^2,"all");
  end % for T
  
  % Output best solution.
  if J < minJ
    minJ = J;
    bestModel = model;
    % Delete iteration counter if present.
    if bscount2 > 0
      outp.info(repmat('\b',1,bscount2));
      bscount2 = 0;
    end
    % Delete previous iteration's output if present.
    if bscount > 0
      outp.info(repmat('\b',1,bscount));
    end
    outp.charcount(0); % start tracking the number of characters output
    outp.info('  MSE  : %-20.4f\n',J);
    if outp.debug
      flatmodel = optutil.flattenstruct(bestModel);
      paramnames = fieldnames(flatmodel);
      paramnamesEact = paramnames(endsWith(paramnames,'_Eact'));
      paramnamesR = paramnames(contains(paramnames,'__R')&~endsWith(paramnames,'_Eact'));
      for pr = 1:length(paramnamesR)
        pname = paramnamesR{pr};
        outp.info('  %-20s:  %10.3g mΩ\n',pname,flatmodel.(pname)*1000);
      end % for
      for pe = 1:length(paramnamesEact)
        pname = paramnamesEact{pe};
        meta = param.modelspec.params.(pname(1:end-5));
        Eact = flatmodel.(pname)/1000;
        EactString = sprintf('%10.3f ',Eact);
        outp.info('  %-20s: %s%s kJ\n',pname,meta.tempcoeff,EactString);
      end % for
    end
    bscount = outp.charcount(); % save number of chars output so we can 
                                % backspace next iteration                       
    
    if outp.plot && ishandle(fig)
      for kt = 1:ntemp
        socPct = param.tempSet(kt).socPct;
        for kz = 1:length(socPct)
          set(lines_ZMOD{kt}(kz),'XData',real(ZMOD{kt}(:,kz)),'Ydata',-imag(ZMOD{kt}(:,kz)));
        end % for soc
      end % for temp
      drawnow;
    end % if
  end % if
  
  % Output optimization metadata.
  if mod(evalnum,10) == 0
    if bscount2 > 0
      outp.info(repmat('\b',1,bscount2));
    end
    outp.charcount(0); % start tracking the number of characters output
    outp.info('  F-evals: %10d\n',evalnum+1);
    bscount2 = outp.charcount(); % save number of chars output so we can 
                            % backspace next iteration
  end
  
  evalnum = evalnum + 1;
  outp.charcount(0); % reset char count
  end % cost()
  
  function J = uicost(model,param)
  %UICOST Cost function for the model regression.
  
  ntemp = length(param.Tvect);
  
  ZLAB = cell(ntemp,1);
  for kt = 1:length(param.Tvect)
      ZLAB{kt} = param.tempSet(kt).Z;
  end
  
  % Iterate temperatures, accumulating cost.
  J = 0;
  ZMOD = getImpedance(model,param);
  for kt = 1:length(param.Tvect)
    % Compute the mean square error between the predicted and actual 
    % admittance.
    err = abs(1./ZLAB{kt}-1./ZMOD{kt})./abs(1./ZLAB{kt});
    J = J + mean(err.^2,"all");
  end % for T
end % uicost()


% GETIMPEDANCE Calculate impedance predicted by TF model.
function [ZMOD, ZNEG, ZPOS, ZEL] = getImpedance(model, param)
  
  R = 8.3144598;   % Molar gas constant [J/mol K]
  F = 96485.3329;  % Faraday constant [C/mol]
  
  ntemp = param.ntemp;
  ZMOD = cell(ntemp,1);
  ZNEG = cell(ntemp,1);
  ZPOS = cell(ntemp,1);
  ZEL = cell(ntemp,1);
  for kt = 1:length(param.Tvect)
    TdegC = param.Tvect(kt);
    paramT = param.tempSet(kt);
    if param.ntemp == 1
      % Single-temperature regression - no activation energies present.
      modelT = model;
    else
      % Evalulate activation energy corrections at this temperature.
      modelT = optutil.evaltemp(model,param.modelspec,TdegC);
    end
    socVect = paramT.socPct;

    % Calculate impedance predicted by the model.
    Zmodel = zeros(size(paramT.Z));
    Zneg = zeros(size(paramT.Z));
    Zpos = zeros(size(paramT.Z));
    Zel = zeros(size(paramT.Z));
    for kz = 1:length(socVect)
      s = 1j*2*pi*paramT.freq(:,kz);

      % Compute impedance predicted by the model at SOC setpoint.
      modelZ = modelT;
      modelZ.const.R   = R;
      modelZ.const.F   = F;
      modelZ.const.T   = TdegC+273.15;
      modelZ.neg.soc   = paramT.thetaNEG(kz);
      modelZ.neg.Uocp  = paramT.UocpNEG(kz);
      modelZ.neg.dUocp = paramT.dUocpNEG(kz);
      modelZ.neg.Ds    = modelZ.neg.Dsref*F/R/(TdegC+273.15)*...
                         modelZ.neg.soc*(modelZ.neg.soc-1)*modelZ.neg.dUocp;
      modelZ.pos.soc   = paramT.thetaPOS(kz);
      modelZ.pos.Uocp  = paramT.UocpPOS(kz);
      modelZ.pos.dUocp = paramT.dUocpPOS(kz);
      modelZ.pos.Ds    = modelZ.pos.Dsref*F/R/(TdegC+273.15)*...
                         modelZ.pos.soc*(modelZ.pos.soc-1)*modelZ.pos.dUocp;
      modelZ.common = []; % force "tfCommon" to recompute 
      [Ctf,Ltf,Jtf,Ztf,Rcttf] = tfCommon(s,modelZ);
      modelZ.common.s = s;
      modelZ.common.C = Ctf;
      modelZ.common.L = Ltf;
      modelZ.common.J = Jtf;
      modelZ.common.Z = Ztf;
      modelZ.common.Rct = Rcttf;
      modelZ.common.ind = ['L1n=1;L2n=2;L1s=3;L1p=4;L2p=5;' ...
        'c1n=1;c2n=2;c3n=3;c4n=4;c1s=5;c2s=6;c1p=7;c2p=8;c3p=9;c4p=10;'...
        'j1n=1;j2n=2;j3n=3;j4n=4;j1p=5;j2p=6;j3p=7;j4p=8;' ...
        'Zsen=1;Zsep=2;Zsn=3;Zsp=4;Isn=5;Isp=6;'];
      Phise = tfPhiseInt(s,[0,3],modelZ);
      Phise0 = Phise(1,:); % - auxPhise.hfGain(1);
      Phise3 = Phise(2,:); % - auxPhise.hfGain(2);
      Phie  = tfPhie(s,3,modelZ);
      Phie3 = Phie; %- auxPhie.hfGain;
      Zmodel(:,kz) = -(Phise3 - Phise0 + Phie3) + modelZ.const.Rc;
      Zneg(:,kz) = Phise0;
      Zpos(:,kz) = -Phise3;
      Zel(:,kz) = -Phie3;
    end % for SOC
    ZMOD{kt} = Zmodel;
    ZNEG{kt} = Zneg;
    ZPOS{kt} = Zpos;
    ZEL{kt} = Zel;
  end % for T
end % getImpedance()


% Utility functions ----------------------------------------------------

%UIEIS Construct functions for handling display of EIS data in the
%  graphical user interface (GUI).
function [initializeFcn, updateFcn] = uiEIS(param)

  % Define globals.
  JPOS = param.modelspec.params.pos__U0.len;
  JNEG = param.modelspec.params.neg__U0.len;
  ntemp = param.ntemp;
  maxnsoc = max(cellfun(@length,{param.tempSet.socPct}));
  Tvect = param.Tvect;
  ZLAB = cell(ntemp,1);
  for kte = 1:ntemp
      ZLAB{kte} = param.tempSet(kte).Z;
  end
  axz = gobjects(1,1);
  axzneg = gobjects(1,1);
  axzpos = gobjects(1,1);
  axzel = gobjects(1,1);
  zplotselect = gobjects(1,1); %#ok<SETNU>
  tempselect = gobjects(1,1); %#ok<SETNU>
  quantityselect = gobjects(1,1); %#ok<SETNU>
  lines_ZMOD = gobjects(maxnsoc,1);
  lines_ZNEG = gobjects(maxnsoc,1);
  lines_ZPOS = gobjects(maxnsoc,1);
  lines_ZEL = gobjects(maxnsoc,1);
  lineRctPOS = gobjects(1,1);
  linesRctjPOS = gobjects(JPOS,1);
  lineRctNEG = gobjects(1,1);
  linesRctjNEG = gobjects(JNEG,1);
  lastParamValues = [];
  [~,tempSelect] = min(abs(Tvect-25));  % choose temp closest to 25degC by default.
  plotSelect = "Nyq";
  quantitySelect = "Z";
  
  initializeFcn = @initializeUIFig;
  updateFcn = @updateUIFig;

  function initializeUIFig(parent)
    TdegC = Tvect(tempSelect);

    % Construct layout.
    gridtop = uigridlayout(parent,[2 1]);
    gridtop.RowHeight = {50, '1x'};
    panelz = uipanel(gridtop,'BorderType','none');
    panelz.Layout.Row = 2;
    panelz.Layout.Column = 1;
    panelcontrols = uipanel(gridtop,'BorderType','none');
    panelcontrols.Layout.Row = 1;
    panelcontrols.Layout.Column = 1;

    % Construct controls.
    gridcontrols = uigridlayout(panelcontrols,[1 3]);
    gridcontrols.RowHeight = {30};
    quantityselect = uidropdown(gridcontrols, ...
        "Items",["Impedance", "Admittance"],...
        "ItemsData",["Z","Y"],"Value",quantitySelect, ...
        "ValueChangedFcn",@(src,event)updateQuantitySelect(event.Value));
    zplotselect = uidropdown(gridcontrols, ...
        "Items",["Nyquist","Bode Magnitude","Bode Phase"], ...
        "ItemsData",["Nyq","BodeMag","BodePhase"],...
        "ValueChangedFcn",@(src,event)updateZPlotType(event.Value));
    tempselect = uidropdown(gridcontrols, ...
        "Items",arrayfun(@(x)sprintf("T=%.1fdegC",x),TdegC), ...
        "ItemsData",1:length(TdegC),"Value",tempSelect,...
        "ValueChangedFcn",@(src,event)updateTempSelect(event.Value));

    % Define plot grid.
    gridplots = uigridlayout(panelz,[3 3]);
    gridplots.RowHeight = {'1x', '1x', '1x'};
    gridplots.ColumnWidth = {'1x', '1x', '1x'};

    % Construct impedance plot.
    axz = uiaxes(gridplots);
    axz.Layout.Row = [1 2];
    axz.Layout.Column = [1 2];
    formatAxes(axz);
    kt = tempSelect;
    socPct = param.tempSet(kt).socPct;
    colors = cool(length(socPct));
    for kz = 1:length(socPct)
      plot(axz, ...
          real(ZLAB{kt}(:,kz)),-imag(ZLAB{kt}(:,kz)), ...
          '.','Color',colors(kz,:)); 
      hold(axz,'on');
    end % for soc
    for kz = 1:length(socPct)
      lines_ZMOD(kz) = plot(axz,NaN,NaN,'-','Color',colors(kz,:));
      hold(axz,'on');
    end % for soc
    xlabel(axz,'$\mathrm{Re}(Z_\mathrm{cell})$ [$\Omega$]','Interpreter','latex');
    ylabel(axz,'$-\mathrm{Im}(Z_\mathrm{cell})$ [$\Omega$]','Interpreter','latex');
    title(axz,'Cell');
    zlab = ZLAB{kt}(:);
    [~,indrm] = rmoutliers(abs(zlab));
    zlab = zlab(~indrm);
    setAxesNyquist('axes',axz,'xdata',real(zlab),'ydata',-imag(zlab));

    % Zneg
    axzneg = uiaxes(gridplots);
    axzneg.Layout.Row = 1;
    axzneg.Layout.Column = 3;
    formatAxes(axzneg);
    kt = tempSelect;
    socPct = param.tempSet(kt).socPct;
    colors = cool(length(socPct));
    for kz = 1:length(socPct)
      lines_ZNEG(kz) = plot(axzneg,NaN,NaN,'-','Color',colors(kz,:));
      hold(axzneg,'on');
    end % for soc
    xlabel(axzneg,'$\mathrm{Re}(Z_\mathrm{neg})$ [$\Omega$]','Interpreter','latex');
    ylabel(axzneg,'$-\mathrm{Im}(Z_\mathrm{neg})$ [$\Omega$]','Interpreter','latex');
    title(axzneg,'Neg. Electrode','Interpreter','latex');

    % Zpos
    axzpos = uiaxes(gridplots);
    axzpos.Layout.Row = 2;
    axzpos.Layout.Column = 3;
    formatAxes(axzpos);
    kt = tempSelect;
    socPct = param.tempSet(kt).socPct;
    colors = cool(length(socPct));
    for kz = 1:length(socPct)
      lines_ZPOS(kz) = plot(axzpos,NaN,NaN,'-','Color',colors(kz,:));
      hold(axzpos,'on');
    end % for soc
    xlabel(axzpos,'$\mathrm{Re}(Z_\mathrm{pos})$ [$\Omega$]','Interpreter','latex');
    ylabel(axzpos,'$-\mathrm{Im}(Z_\mathrm{pos})$ [$\Omega$]','Interpreter','latex');
    title(axzpos,'Pos. Electrode','Interpreter','latex');

    % Zel
    axzel = uiaxes(gridplots);
    axzel.Layout.Row = 3;
    axzel.Layout.Column = 3;
    formatAxes(axzel);
    kt = tempSelect;
    socPct = param.tempSet(kt).socPct;
    colors = cool(length(socPct));
    for kz = 1:length(socPct)
      lines_ZEL(kz) = plot(axzel,NaN,NaN,'-','Color',colors(kz,:));
      hold(axzel,'on');
    end % for soc
    xlabel(axzel,'$\mathrm{Re}(Z_\mathrm{el})$ [$\Omega$]','Interpreter','latex');
    ylabel(axzel,'$-\mathrm{Im}(Z_\mathrm{el})$ [$\Omega$]','Interpreter','latex');
    title(axzel,'Electrolyte','Interpreter','latex');

    % Construct Rct plots.
    % POS
    axtmp = uiaxes(gridplots);
    axtmp.Layout.Row = 3;
    axtmp.Layout.Column = 2;
    formatAxes(axtmp);
    for k = 1:length(linesRctjPOS)
      linesRctjPOS(k) = semilogy(axtmp,NaN,NaN,':');
      hold(axtmp,'on');
    end
    lineRctPOS(1) = semilogy(axtmp,NaN,NaN,'k-');
    hold(axtmp,'on');
    xlabel(axtmp,'SOC, z [\%]','Interpreter','latex');
    ylabel(axtmp,'$R_{ct}^p$','Interpreter','latex');
    title(axtmp,'$R_{ct}^p$ vs SOC','Interpreter','latex');
    % NEG
    axtmp = uiaxes(gridplots);
    axtmp.Layout.Row = 3;
    axtmp.Layout.Column = 1;
    formatAxes(axtmp);
    for k = 1:length(linesRctjNEG)
      linesRctjNEG(k) = semilogy(axtmp,NaN,NaN,':');
      hold(axtmp,'on');
    end
    lineRctNEG(1) = semilogy(axtmp,NaN,NaN,'k-');
    hold(axtmp,'on');
    xlabel(axtmp,'SOC, z [\%]','Interpreter','latex');
    ylabel(axtmp,'$R_{ct}^n$','Interpreter','latex');
    title(axtmp,'$R_{ct}^n$ vs SOC','Interpreter','latex');

    drawnow;
  end % initializeUIPlot()
    
  function updateUIFig(paramValues)
    % Save parameter values so we can update the plots when plot type
    % changes.
    lastParamValues = paramValues;
    nsoc = length(param.tempSet(tempSelect).socPct);
    freq = param.tempSet(tempSelect).freq(:,1);
    TdegC = Tvect(tempSelect);

    % Calculate impedance predicted by the TF model.
    [Zmodel, Zneg, Zpos, Zel] = getImpedance(paramValues,param);
    Zmodel = Zmodel{tempSelect};
    Zneg = Zneg{tempSelect};
    Zpos = Zpos{tempSelect};
    Zel = Zel{tempSelect};
    if quantitySelect == "Y"
      Zmodel = 1./Zmodel;
      Zneg = 1./Zneg;
      Zpos = 1./Zpos;
      Zel = 1./Zel;
    end

    % Calculate Rct.
    socPct = linspace(0,100,100);
    % POS
    theta0 = paramValues.pos.theta0;
    theta100 = paramValues.pos.theta100;
    t = theta0 + (socPct/100)*(theta100-theta0);
    ocpmodel = MSMR(paramValues.pos);
    ctDataPOS = ocpmodel.Rct(paramValues.pos,'theta',t,'TdegC',TdegC);
    % NEG
    theta0 = paramValues.neg.theta0;
    theta100 = paramValues.neg.theta100;
    t = theta0 + (socPct/100)*(theta100-theta0);
    ocpmodel = MSMR(paramValues.neg);
    ctDataNEG = ocpmodel.Rct(paramValues.neg,'theta',t,'TdegC',TdegC);

    % Update model predictions on plots.
    if plotSelect == "Nyq"
      for idxSOC = 1:nsoc
        lines_ZMOD(idxSOC).XData = real(Zmodel(:,idxSOC));
        lines_ZMOD(idxSOC).YData = -imag(Zmodel(:,idxSOC));
        lines_ZNEG(idxSOC).XData = real(Zneg(:,idxSOC));
        lines_ZNEG(idxSOC).YData = -imag(Zneg(:,idxSOC));
        lines_ZPOS(idxSOC).XData = real(Zpos(:,idxSOC));
        lines_ZPOS(idxSOC).YData = -imag(Zpos(:,idxSOC));
        lines_ZEL(idxSOC).XData = real(Zel(:,idxSOC));
        lines_ZEL(idxSOC).YData = -imag(Zel(:,idxSOC));
      end
      setAxesNyquist('axes',axzneg);
      setAxesNyquist('axes',axzpos);
      setAxesNyquist('axes',axzel);
    elseif plotSelect == "BodeMag"
      for idxSOC = 1:nsoc
        lines_ZMOD(idxSOC).XData = freq;
        lines_ZMOD(idxSOC).YData = abs(Zmodel(:,idxSOC));
        lines_ZNEG(idxSOC).XData = freq;
        lines_ZNEG(idxSOC).YData = abs(Zneg(:,idxSOC));
        lines_ZPOS(idxSOC).XData = freq;
        lines_ZPOS(idxSOC).YData = abs(Zpos(:,idxSOC));
        lines_ZEL(idxSOC).XData = freq;
        lines_ZEL(idxSOC).YData = abs(Zel(:,idxSOC));
        xlim(axzneg,[min(freq) max(freq)]);
        xlim(axzpos,[min(freq) max(freq)]);
        xlim(axzel,[min(freq) max(freq)]);
        ylim(axzneg,[min(abs(Zneg(:,idxSOC))) max(abs(Zneg(:,idxSOC)))]);
        ylim(axzpos,[min(abs(Zpos(:,idxSOC))) max(abs(Zpos(:,idxSOC)))]);
        ylim(axzel,[min(abs(Zel(:,idxSOC))) max(abs(Zel(:,idxSOC)))]);
      end
    elseif plotSelect == "BodePhase"
      for idxSOC = 1:nsoc
        lines_ZMOD(idxSOC).XData = freq;
        lines_ZMOD(idxSOC).YData = angle(Zmodel(:,idxSOC))*180/pi;
        lines_ZNEG(idxSOC).XData = freq;
        lines_ZNEG(idxSOC).YData = angle(Zneg(:,idxSOC))*180/pi;
        lines_ZPOS(idxSOC).XData = freq;
        lines_ZPOS(idxSOC).YData = angle(Zpos(:,idxSOC))*180/pi;
        lines_ZEL(idxSOC).XData = freq;
        lines_ZEL(idxSOC).YData = angle(Zel(:,idxSOC))*180/pi;
        xlim(axzneg,[min(freq) max(freq)]);
        xlim(axzpos,[min(freq) max(freq)]);
        xlim(axzel,[min(freq) max(freq)]);
        ylim(axzneg,[min(angle(Zneg(:,idxSOC))*180/pi) max(angle(Zneg(:,idxSOC))*180/pi)]);
        ylim(axzpos,[min(angle(Zpos(:,idxSOC))*180/pi) max(angle(Zpos(:,idxSOC))*180/pi)]);
        ylim(axzel,[min(angle(Zel(:,idxSOC))*180/pi) max(angle(Zel(:,idxSOC))*180/pi)]);
      end
    elseif plotSelect == "Real"
      for idxSOC = 1:nsoc
        lines_ZMOD(idxSOC).XData = freq;
        lines_ZMOD(idxSOC).YData = real(Zmodel(:,idxSOC));
      end
    elseif plotSelect == "Imag"
      for idxSOC = 1:nsoc
        lines_ZMOD(idxSOC).XData = freq;
        lines_ZMOD(idxSOC).YData = imag(Zmodel(:,idxSOC));
      end
    end

    % Update Rct plots.
    lineRctPOS.XData = socPct;
    lineRctPOS.YData = ctDataPOS.Rct;
    for j = 1:JPOS
      linesRctjPOS(j).XData = socPct;
      linesRctjPOS(j).YData = ctDataPOS.Rctj(j,:);
    end
    lineRctNEG.XData = socPct;
    lineRctNEG.YData = ctDataNEG.Rct;
    for j = 1:JNEG
      linesRctjNEG(j).XData = socPct;
      linesRctjNEG(j).YData = ctDataNEG.Rctj(j,:);
    end
  end % updateUIFIG()
    
  function updateZPlotType(plottype)
    plotSelect = plottype;
    redrawPlots();
  end
  
  function updateTempSelect(indT)
    tempSelect = indT;
    redrawPlots();
  end

  function updateQuantitySelect(qty)
    quantitySelect = qty;
    redrawPlots();
  end
    
  function redrawPlots()
    Zlab = ZLAB{tempSelect};
    if quantitySelect == "Y"
      Zlab = 1./Zlab;
    end

    if plotSelect == "Nyq"  
      cla(axz); cla(axzneg); cla(axzpos); cla(axzel);
      axz.XScale = 'linear';
      axz.YScale = 'linear';
      axzneg.XScale = 'linear';
      axzneg.YScale = 'linear';
      axzpos.XScale = 'linear';
      axzpos.YScale = 'linear';
      axzel.XScale = 'linear';
      axzel.YScale = 'linear';
      kt = tempSelect;
      socPct = param.tempSet(kt).socPct;
      colors = cool(length(socPct));
      for kz = 1:length(socPct)
        plot(axz, ...
            real(Zlab(:,kz)),-imag(Zlab(:,kz)), ...
            '.','Color',colors(kz,:)); 
        hold(axz,'on');
      end % for soc
      for kz = 1:length(socPct)
        lines_ZMOD(kz) = plot(axz,NaN,NaN,'-','Color',colors(kz,:));
        lines_ZNEG(kz) = plot(axzneg,NaN,NaN,'-','Color',colors(kz,:));
        lines_ZPOS(kz) = plot(axzpos,NaN,NaN,'-','Color',colors(kz,:));
        lines_ZEL(kz) = plot(axzel,NaN,NaN,'-','Color',colors(kz,:));
        hold(axz,'on');
        hold(axzneg,'on');
        hold(axzpos,'on');
        hold(axzel,'on');
      end % for soc
      xlabel(axz,['$\mathrm{Re}(' char(quantitySelect) '_\mathrm{cell})$'],'Interpreter','latex');
      ylabel(axz,['$-\mathrm{Im}(' char(quantitySelect) '_\mathrm{cell})$'],'Interpreter','latex');
      zlab = Zlab(:);
      [~,indrm] = rmoutliers(abs(zlab));
      zlab = zlab(~indrm);
      setAxesNyquist('axes',axz,'xdata',real(zlab),'ydata',-imag(zlab));
      xlabel(axzneg,['$\mathrm{Re}(' char(quantitySelect) '_\mathrm{neg})$'],'Interpreter','latex');
      ylabel(axzneg,['$-\mathrm{Im}(' char(quantitySelect) '_\mathrm{neg})$'],'Interpreter','latex');
      xlabel(axzpos,['$\mathrm{Re}(' char(quantitySelect) '_\mathrm{pos})$'],'Interpreter','latex');
      ylabel(axzpos,['$-\mathrm{Im}(' char(quantitySelect) '_\mathrm{pos})$'],'Interpreter','latex');
      xlabel(axzel,['$\mathrm{Re}(' char(quantitySelect) '_\mathrm{el})$'],'Interpreter','latex');
      ylabel(axzel,['$-\mathrm{Im}(' char(quantitySelect) '_\mathrm{el})$'],'Interpreter','latex');
    elseif plotSelect == "BodeMag"
      cla(axz); cla(axzneg); cla(axzpos); cla(axzel);
      axz.XScale = 'log';
      axz.YScale = 'log';
      axzneg.XScale = 'log';
      axzneg.YScale = 'log';
      axzpos.XScale = 'log';
      axzpos.YScale = 'log';
      axzel.XScale = 'log';
      axzel.YScale = 'log';
      kt = tempSelect;
      socPct = param.tempSet(kt).socPct;
      freq = param.tempSet(kt).freq;
      colors = cool(length(socPct));
      for kz = 1:length(socPct)
        plot(axz, ...
            freq(:,kz),abs(Zlab(:,kz)), ...
            '.','Color',colors(kz,:)); 
        hold(axz,'on');
      end % for soc
      for kz = 1:length(socPct)
        lines_ZMOD(kz) = plot(axz,NaN,NaN,'-','Color',colors(kz,:)); 
        lines_ZNEG(kz) = plot(axzneg,NaN,NaN,'-','Color',colors(kz,:)); 
        lines_ZPOS(kz) = plot(axzpos,NaN,NaN,'-','Color',colors(kz,:)); 
        lines_ZEL(kz) = plot(axzel,NaN,NaN,'-','Color',colors(kz,:));
        hold(axz,'on');
        hold(axzneg,'on');
        hold(axzpos,'on');
        hold(axzel,'on');
      end % for soc
      xlim(axz,[min(freq,[],"all") max(freq,[],"all")]);
      ylim(axz,[min(abs(Zlab),[],"all") max(abs(Zlab),[],"all")]);
      xlabel(axz,'Cyclic frequency, $f$ [Hz]','Interpreter','latex');
      ylabel(axz,['$|' char(quantitySelect) '_{cell}(j2\pi f)|$'],'Interpreter','latex');
      title(axz,sprintf('%.2fdegC',Tvect(kt)));
      xlabel(axzneg,'Cyclic frequency, $f$ [Hz]','Interpreter','latex');
      ylabel(axzneg,['$|' char(quantitySelect) '_{neg}(j2\pi f)|$'],'Interpreter','latex');
      xlabel(axzpos,'Cyclic frequency, $f$ [Hz]','Interpreter','latex');
      ylabel(axzpos,['$|' char(quantitySelect) '_{pos}(j2\pi f)|$'],'Interpreter','latex');
      xlabel(axzel,'Cyclic frequency, $f$ [Hz]','Interpreter','latex');
      ylabel(axzel,['$|' char(quantitySelect) '_{el}(j2\pi f)|$'],'Interpreter','latex');
    elseif plotSelect == "BodePhase"
      cla(axz); cla(axzneg); cla(axzpos); cla(axzel);
      axz.XScale = 'log';
      axz.YScale = 'linear';
      axzneg.XScale = 'log';
      axzneg.YScale = 'linear';
      axzpos.XScale = 'log';
      axzpos.YScale = 'linear';
      axzel.XScale = 'log';
      axzel.YScale = 'linear';
      kt = tempSelect;
      socPct = param.tempSet(kt).socPct;
      freq = param.tempSet(kt).freq;
      colors = cool(length(socPct));
      for kz = 1:length(socPct)
        plot(axz, ...
            freq(:,kz),angle(Zlab(:,kz))*180/pi, ...
            '.','Color',colors(kz,:)); 
        hold(axz,'on');
      end % for soc
      for kz = 1:length(socPct)
        lines_ZMOD(kz) = plot(axz,NaN,NaN,'-','Color',colors(kz,:)); 
        lines_ZNEG(kz) = plot(axzneg,NaN,NaN,'-','Color',colors(kz,:)); 
        lines_ZPOS(kz) = plot(axzpos,NaN,NaN,'-','Color',colors(kz,:)); 
        lines_ZEL(kz) = plot(axzel,NaN,NaN,'-','Color',colors(kz,:));
        hold(axz,'on');
        hold(axzneg,'on');
        hold(axzpos,'on');
        hold(axzel,'on');
      end % for soc
      xlim(axz,[min(freq,[],"all") max(freq,[],"all")]);
      ylim(axz,[min(angle(Zlab)*180/pi,[],"all") max(angle(Zlab)*180/pi,[],"all")]);
      xlabel(axz,'Cyclic frequency, $f$ [Hz]','Interpreter','latex');
      ylabel(axz,['$\angle ' char(quantitySelect) '_{cell}(j2\pi f)$ [$\circ$]'],'Interpreter','latex');
      title(axz,sprintf('%.2fdegC',Tvect(kt)));
      xlabel(axzneg,'Cyclic frequency, $f$ [Hz]','Interpreter','latex');
      ylabel(axzneg,['$\angle ' char(quantitySelect) '_{neg}(j2\pi f)$ [$\circ$]'],'Interpreter','latex');
      xlabel(axzpos,'Cyclic frequency, $f$ [Hz]','Interpreter','latex');
      ylabel(axzpos,['$\angle ' char(quantitySelect) '_{pos}(j2\pi f)$ [$\circ$]'],'Interpreter','latex');
      xlabel(axzel,'Cyclic frequency, $f$ [Hz]','Interpreter','latex');
      ylabel(axzel,['$\angle ' char(quantitySelect) '_{el}(j2\pi f)$ [$\circ$]'],'Interpreter','latex');
    elseif plotSelect == "Real"
      cla(axz);
      axz.XScale = 'log';
      axz.YScale = 'linear';
    elseif plotSelect == "Imag"
      cla(axz);
      axz.XScale = 'log';
      axz.YScale = 'linear'; 
    end
    updateUIFig(lastParamValues);
  end % redrawPlots()
end % uiEIS()

function ax = formatAxes(ax)
  ax.PlotBoxAspectRatioMode = 'manual';
  ax.PlotBoxAspectRatio = [2/(sqrt(5)-1) 1 1];
  ax.LineWidth = 1;
  ax.FontName = 'Times';
  ax.FontSize = 11;
  ax.TitleFontWeight = 'normal';
  ax.TitleFontSizeMultiplier = 1.2;
  ax.LabelFontSizeMultiplier = 1.1;
  ax.XMinorTick = 'on';
  ax.YMinorTick = 'on';
end