% FITDISCHG Regress an SPM with composition-dependent diffusivity to cell
%  discharge or charge measurements.
%
% -- Usage --
% outData = FITDISCHG(cellspec,configREG,direction)
% outData = FITDISCHG(cellspec,configREG,direction,narrowing)
% outData = FITDISCHG(...,'ocpRES',ocpres)
%
% -- Input --
% cellspec    = cell specification generated by labcell() function
% configREG   = structure of configuration options for the model 
%               regression (see fields below)
% direction   = 'dis' or 'chg' (regress to discharge or charge data)
% ocpres      = number of points to use to approximate OCP curves of the
%               electrodes
%
% The fields of the configREG struct are:
%   .init  : Structure of initial values for mPOS, bPOS, 
%            DsrefPOS, and DsrefNEG
%   .lb    : Structure of lower bounds.
%   .ub    : Structure of upper bounds.
%   .pin   : Structure of parameter values to pin (i.e., to not
%               optimize over). OPTIONAL.
%
% The optional argument, `narrowing`, is a string with any of the following 
% three parts separated by whitespace. All of these are optional and may
% appear in any order.
%  - TdegC : A string of the form 'XXdegC' where XX is the temperature in
%            in degrees Celcius. (e.g., '25degC' and '-15degC' are valid.)
%  - rate  : A string of the form 'XXC' where XX is the C-rate.
%            (e.g., '1C' and '0.5C' are valid but 'C/2' is not.)
%  - cellID: A string specifying the ID of the cell.
%            (e.g., 'FP41'.)
% Examples: '25degC 1C', '1C', 'FP41 0.5C' are all valid narrowing strings.
%
% -- Known issues --
% * Stability problems due to variation in Ds with lithium composition.
%   TODO: use an adaptive sampling interval to overcome this problem
%
% * MSMR regression is not the most robust
%   TODO: use pso with fmincon to increase robustness to local minima
%
% * Magic numbers:
%    - @ computation of R0 inside cost function
%    - @ upper/lower bounds of MSMR regression
%   TODO: remove these magic numbers
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         
function outData = fitDischg(cellspec,configREG,direction,varargin)

  istemp = @(x)(ischar(x)&&endsWith(x,'degC')&& ~isnan(str2double(x(1:end-4))));
  israte = @(x)(ischar(x)&&endsWith(x,'C')&&~isnan(str2double(x(1:end-1))));
  parser = inputParser;
  parser.addRequired('cellspec',@(x)isstruct(x)&&strcmp(x.origin__,'labcell'));
  parser.addRequired('configREG',@(x)isscalar(x)&&isstruct(x));
  parser.addRequired('direction',@(x)ischar(x)&&any(strcmp(x,{'dis','chg'})));
  parser.addOptional('narrowing',[],@(x)ischar(x));
  parser.addParameter('pin',struct,@(x)isstruct(x)&&isscalar(x));
  parser.addParameter('ocpRES',1000,@(x)isnumeric(x)&&isscalar(x)&&x>1);
  parser.addParameter('usePosMSMR',true,@(x)isscalar(x)&&islogical(x));
  parser.parse(cellspec,configREG,direction,varargin{:});
  arg = parser.Results; % struct of validated arguments
  
  % Collect cell metadata.
  cellname = arg.cellspec.name;
  processfile = arg.cellspec.dis.processfile;
  fitfile = arg.cellspec.dis.fitfile;
  timestamp = arg.cellspec.timestamp__;
  
  outp.print('Started fitDischg %s\n',cellname);
  
  % Load processed discharge data.
  processData = load(processfile);
  cal = processData.cal;           % struct array of calibrated dischg data
  cellData = processData.cellData; % struct of param values previously identified
  dataOCV = processData.dataOCV;   % raw data from OCV regression
  dataOCP = processData.dataOCP;   % raw data from OCP regression
  
  % Select the dataset based on the narrowing parameters.
  if isempty(arg.narrowing)
    matches = cal;
  else
    specs = split(arg.narrowing);

    % Determine narrowing parameters.
    TdegC = [];
    C_rate = [];
    cellID = [];
    for k = 1:length(specs)
      spec = specs{k};
      if istemp(spec)
        TdegC = str2double(spec(1:end-4));
      elseif israte(spec)
        C_rate = str2double(spec(1:end-1));
      else
        cellID = spec;
      end
    end % for

    % Search for matching datasets.
    clear matches;
    cursor = length(cal);
    for k = length(cal):-1:1
      if ~isempty(TdegC) && cal(k).TdegC ~= TdegC
        continue;
      end
      if ~isempty(C_rate) && cal(k).C_rate ~= C_rate
        continue;
      end
      if ~isempty(cellID) && ~strcmp(cal(k).cellID,cellID)
        continue;
      end

      % We have a match!
      matches(cursor) = cal(k);
      cursor = cursor - 1;
    end
    matches = matches(cursor+1:end);  % delete unfilled elements
    if isempty(matches)
      error('Couldn''t find any tests for ''%s''. Cannot continue.', ...
          arg.narrowing);
    end
  end % if
  
  % If multiple matches, then ask the user to select which dataset to use.
  if length(matches) > 1
    N = length(matches);
    outp.required('Multiple matching datasets! Please disambiguate.\n');
    outp.required('  %-2s %-7s %-7s %-7s %-7s\n', ...
        '#','Cell','TdegC','C-rate','Q [Ah]');
    for k = 1:N
      dat = matches(k);
      outp.required('  %-2d %-7s %-7.2f %-7.2f %-7.2f\n', ...
          k,dat.cellID,dat.TdegC,dat.C_rate,dat.QAh);
    end
    selector = -1;
    while (~isnumeric(selector) || ~isscalar(selector) || selector < 1 || selector > N)
      commandwindow; % bring focus to command window
      outp.required('Enter dataset number (1-%d) > ',N);
      selector = input('');
    end
    dataset = matches(selector);
  else
    dataset = matches(1);
  end
  
  outp.info('INFO: Using %s %.2fdegC %.2fC %.2fAh dataset.\n', ...
      dataset.cellID,dataset.TdegC,dataset.C_rate,dataset.QAh);
  
  % Fetch test parameters and previously-identified cell parameter values at
  % the same temperature.
  % Store all values in parameter struct `p`.
  p = struct;
  p.TdegC = dataset.TdegC;           % test temperature
  p.C_rate = dataset.C_rate_actual;  % actual C-rate
  indT = find(cellData.const.TdegC==p.TdegC,1,'first');
  if isempty(indT)
    error(['Could not match test @ %.0fdegC to OCV test at same ' ...
        'temperature. Cannot continue.'], p.TdegC);
  end
  p.QAh = cellData.const.Q(indT);
  
  % Fetch OCP curves.
  % NEG - use regressed MSMR model (absolute OCP)
  p.neg = struct;
  p.neg.U0 = cellData.neg.U0(:,indT);
  p.neg.X = cellData.neg.X(:,indT);
  p.neg.omega = cellData.neg.omega(:,indT);
  p.neg.theta0 = cellData.neg.theta0(:,indT);
  p.neg.theta100 = cellData.neg.theta100(:,indT);
  thetaNEG = linspace(0.001,0.999,arg.ocpRES);
  ocpNEG = MSMR(p.neg).ocp('theta',thetaNEG,'TdegC',TdegC);
  p.neg.theta = thetaNEG;
  p.neg.Uocp = ocpNEG.Uocp;
  p.neg.dUocp = ocpNEG.dUocp;
  
  if arg.usePosMSMR
    % POS - move regressed MSMR model over relative stiochiometry
    p.pos = struct;
    p.pos.U0 = cellData.pos.U0(:,indT);
    p.pos.X = cellData.pos.X(:,indT);
    p.pos.omega = cellData.pos.omega(:,indT);
    p.pos.theta0 = cellData.pos.theta0(:,indT);
    p.pos.theta100 = cellData.pos.theta100(:,indT);
    thetaPOS = linspace(max(0.001,p.pos.theta100),min(0.999,p.pos.theta0),arg.ocpRES);
    ocpPOS = MSMR(p.pos).ocp('theta',thetaPOS,'TdegC',TdegC);
    p.pos.theta0Tilde = 1;
    p.pos.theta100Tilde = 0;
    p.pos.thetaTilde = (thetaPOS-p.pos.theta100)/(p.pos.theta0-p.pos.theta100);
    p.pos.UocpTilde = ocpPOS.Uocp;
    p.pos.dUocpTilde = ocpPOS.dUocp*(p.pos.theta0-p.pos.theta100);
  else
    % POS - use relative OCP curve.
    p.pos.theta0Tilde = max(dataOCV.ocprelPOS.Z);
    p.pos.theta100Tilde = min(dataOCV.ocprelPOS.Z);
    p.pos.thetaTilde = dataOCV.ocprelPOS.Z;
    p.pos.UocpTilde = dataOCV.ocprelPOS.U;
    p.pos.dUocpTilde = roughdiff(dataOCV.ocprelPOS.U,dataOCV.ocprelPOS.Z);
  end
  
  % Plot OCP curve for POS.
  if outp.debug
    figure;
    plot(p.pos.thetaTilde,p.pos.UocpTilde);
    xlabel('Relative composition, $\tilde{\theta}_s$','Interpreter','latex');
    ylabel('OCP, $U_\mathrm{ocp}$ [V]','Interpreter','latex');
    title('OCP: Positive Electrode');
    thesisFormat;
  end
  
  % Fetch configuration data.
  configREG = evalstrings(arg.configREG,p);
  p.ts = configREG.ts;
  p.rTildeRef = configREG.rTildeRef;
  if isfield(configREG,'pin')
    p.pin = configREG.pin;
  else
    p.pin = struct;
  end
  
  % Fetch time-domain measurements and resample with uniform sample interval.
  timeLAB = dataset.([arg.direction 'T']);
  iappLAB = dataset.([arg.direction 'It']);
  vcellLAB = dataset.([arg.direction 'Vt']);
  timeUniform = 0:p.ts:max(timeLAB);
  p.iappLAB = linearinterp(timeLAB,iappLAB,timeUniform);
  p.vcellLAB = linearinterp(timeLAB,vcellLAB,timeUniform);
  p.timeLAB = timeUniform;
  
  % Set initial SOC of the cell.
  if strcmp(arg.direction,'chg')
    p.soc0 = 0;  % start from 0% SOC
  else
    p.soc0 = 1;  % start from 100% SOC
  end 
  
  % Configure and run optimizer.
  init = stuff(configREG.init,p.pin);  % initial vector
  lb = stuff(configREG.lb,p.pin);      % vector of lower bounds
  ub = stuff(configREG.ub,p.pin);      % vector of upper bounds
  Np = configREG.Np;             % number of particles
  pop0 = init(:).';              % (partial) initial population
  optionsFMINCON = optimoptions(@fmincon,'Display','off',...
    'MaxFunEvals',1e6,'MaxIter',1e3,...
    'TolFun',1e-20,'TolX',1e-20,'TolCon',1e-20);
  optionsPSO = optimoptions(@particleswarm,...
    'Display','off','UseParallel',false,...
    'FunctionTolerance',1e-20,'SwarmSize',Np,...
    'MaxIterations',500*length(init),'MaxStallIterations',100,...
    'InitialSwarmMatrix',pop0,'FunValCheck','off',...
    'MaxTime',20*1,'HybridFcn',{@fmincon,optionsFMINCON});
  outp.info('Running dis/charge regression...\n');
  vect = particleswarm(@(x)cost(x,p),length(init),lb,ub,optionsPSO);
  params = explode(vect,p.pin);
  outp.info('  DsrefNEG       : ');
  outp.info('%5.3e ',params.DsrefNEG);
  outp.ln;
  outp.info('  DsrefPOS       : ');
  outp.info('%5.3e ',params.DsrefPOS);
  outp.ln;
  outp.info('  mPOS           : ');
  outp.info('%5.3f ',params.mPOS);
  outp.ln;
  outp.info('  bPOS           : ');
  outp.info('%5.3f ',params.bPOS);
  outp.ln;
  outp.info('finished regression\n');
  
  % Collect cell parameters at test temperature.
  secnames = fieldnames(cellData);
  for ks = 1:length(secnames)
    secname = secnames{ks};
    sec = cellData.(secname);
    paramnames = fieldnames(sec);
    for kp = 1:length(paramnames)
      paramname = paramnames{kp};
      val = cellData.(secname).(paramname);
      cellData.(secname).(paramname) = val(:,indT);
    end % for
  end % for
  cellData.neg.Dsref = params.DsrefNEG;
  cellData.pos.Dsref = params.DsrefPOS;
  
  estimators = fieldnames(dataOCP.ocpfitPOS);
  estimator = estimators{1}; % use results of first OCP estimator
  ocpfitPOS = [dataOCP.ocpfitPOS.(estimator)];
  indT = p.TdegC == [ocpfitPOS.TdegC];
  ocpfitPOS = ocpfitPOS(indT);
  
  % Compute absolute stoichiometry limits for POS and re-run MSMR model
  % regression to determine U0, X, omega.
  outp.info('Updating MSMR parameters...\n');
  m = params.mPOS;
  b = params.bPOS;
  cellData.pos.theta0 = 1 - (m/b)*(b-p.pos.theta0Tilde);
  cellData.pos.theta100 = 1 - (m/b)*(b-p.pos.theta100Tilde);
  ocpestPOS = struct;
  ocpestPOS.Z = p.pos.thetaTilde;
  ocpestPOS.U = p.pos.UocpTilde;
  ocpestPOS.dZ = 1./p.pos.dUocpTilde;
  ocpestPOS.name = 'dischg';
  ocpestPOS.elecname = 'POS';
  ocpestPOS.TdegC = p.TdegC;
  configPOS = dataOCP.arg__.configPOS;
  configPOS.init = ocpfitPOS.params;
  configPOS.init.thetamin = cellData.pos.theta100;
  configPOS.lb.thetamin = cellData.pos.theta100;
  configPOS.ub.thetamin = cellData.pos.theta100;
  configPOS.init.thetamax = cellData.pos.theta0;
  configPOS.lb.thetamax = cellData.pos.theta0;
  configPOS.ub.thetamax = cellData.pos.theta0;
  configPOS.lb.U0 = configPOS.init.U0*0.9;
  configPOS.lb.X = configPOS.init.X*0.5;
  configPOS.lb.omega = configPOS.init.omega*0.5;
  configPOS.ub.U0 = configPOS.init.U0/0.9;
  configPOS.ub.X = configPOS.init.X/0.5;
  configPOS.ub.omega = configPOS.init.omega/0.5;
  fitPOS = fitMSMR(ocpestPOS,configPOS);
  cellData.pos.U0 = fitPOS.params.U0(:);
  cellData.pos.X = fitPOS.params.X(:);
  cellData.pos.omega = fitPOS.params.omega(:);
  
  % Collect and save output data.
  outData = struct;
  outData.cellData = cellData;
  outData.params = params;
  outData.origin__ = 'fitDischg';
  outData.arg__ = arg;
  outData.timestamp__ = timestamp;
  save(fitfile,'-struct','outData');
  
  outp.print('Finished fitDischg %s\n',cellname);
  
end
  
  
% Utility functions -------------------------------------------------------

% COST Cost function for the model regression.
function J = cost(vect,arg)
  
  persistent minJ fig line_vcellLAB line_vcellMOD;
  if isempty(minJ)
    minJ = Inf;
  end
  if outp.plot && (isempty(fig) || ~ishandle(fig))
    minJ = Inf;
    fig = figure('Name','Discharge Regression','WindowStyle','docked');
    for k = 1:2
      line_vcellLAB = plot(NaN,NaN,'k:'); hold on;
      line_vcellMOD = plot(NaN,NaN,'m');
      xlabel('Time, $t$ [min]', ...
          'Interpreter','latex');
      ylabel('$v_\mathrm{cell}$ [$\mathrm{V}$]','Interpreter','latex');
      title('SPM Regression');
      legend('Lab','Model','Location','north');
      thesisFormat;
      drawnow;
    end
  end
  
  % Fetch optimization parameter values.
  params = explode(vect,arg.pin);
  arg.params = params;
  
  % Compute prediction of SPM without series resistance (-iapp*R0) term.
  vcellMODWithoutR0 = runSPM(arg.iappLAB,arg.timeLAB,arg.soc0,arg);
  vcellLAB = arg.vcellLAB; % true (measured) cell voltage
  
  % Approximate series resistance (R0) by least squares linear regression.
  ind = vcellLAB>=2.95;
  R0 = lsqnonneg(-arg.iappLAB(ind),vcellLAB(ind)-vcellMODWithoutR0(ind));
  
  % Estimate cell voltage with R0 term.
  vcellMOD = vcellMODWithoutR0 - arg.iappLAB*R0;
  
  % Compute the RMS error between the predicted and actual voltages.
  J = rms(vcellLAB - vcellMOD);
  
  if J < minJ
    if ~isinf(minJ)
      outp.info(repmat('\b',1,length('  RMSE  : xxxxxxxxxxxxxxxxxxxxx mV')));
    end
    outp.info('  RMSE  : %-20.3f mV\n',1000*J);
    minJ = J;
    
    if outp.plot && ishandle(fig)
      set(line_vcellLAB,'XData',arg.timeLAB/60,'Ydata',vcellLAB);
      set(line_vcellMOD,'XData',arg.timeLAB/60,'Ydata',vcellMOD);
      drawnow;
    end
  end
end

% RUNSPM Simulate voltage prediction of the single-particle model (SPM).
% !! The output voltage (vcell) does not include the series resistance 
%    (-iapp*R0) term. You must add this term manually after calling 
%    this function.
function vcell = runSPM(iapp,time,soc0,arg)
  % Assign constants.
  F = 96485.3365;           % Faraday's constant [C mol^-1]
  R = 8.3144621;            % Universal gas constant [J mol^-1K^-1]
  T = arg.TdegC+273.15;     % Temperature [K]
  ts = time(2)-time(1);     % Sampling interval [s].
  
  % Assign electrochemical properties.
  theta0NEG        = arg.neg.theta0;
  thetaTilde0POS   = arg.pos.theta0Tilde;
  theta100NEG      = arg.neg.theta100;
  thetaTilde100POS = arg.pos.theta100Tilde;
  dthetaNEG        = theta100NEG - theta0NEG;
  dthetaTildePOS   = thetaTilde100POS - thetaTilde0POS;
  UocpNEG       = @(theta)linearinterp(arg.neg.theta,arg.neg.Uocp,theta);
  UocpTildePOS  = @(thTilde)linearinterp(arg.pos.thetaTilde,arg.pos.UocpTilde,thTilde);
  dUocpNEG      = @(theta)linearinterp(arg.neg.theta,arg.neg.dUocp,theta);
  dUocpTildePOS = @(thTilde)linearinterp(arg.pos.thetaTilde,arg.pos.dUocpTilde,thTilde);
  QAh           = arg.QAh;
  ThQNEG        = abs(dthetaNEG)/(10800*QAh);
  ThQPOS        = abs(dthetaTildePOS)/(10800*QAh);
  rTildeRef     = arg.rTildeRef; % reference radius for Ds calculation
  DsrefNEG      = arg.params.DsrefNEG;
  DsrefPOS      = arg.params.DsrefPOS;
  m             = arg.params.mPOS;
  b             = arg.params.bPOS;
  
  % Compute initial stoichiometries of the electrodes (relative for POS).
  thetaNEG0 = theta0NEG + soc0*dthetaNEG;
  thetaTildePOS0 = thetaTilde0POS + soc0*dthetaTildePOS;
  
  % Generate constant state-space matrices for NEG and POS.
  CNEG = [1, 8/35];
  CPOS = [1, 8/35];
  
  % Initialize simulation variables.
  DsNEG = DsrefNEG*(F/(R*T))*thetaNEG0*...
    (thetaNEG0-1)*dUocpNEG(thetaNEG0);
  DsPOS = DsrefPOS*(F/(R*T))*(1-(m/b)*(b-thetaTildePOS0))*...
    (thetaTildePOS0-b)*dUocpTildePOS(thetaTildePOS0);
  thetaAvgNEG = thetaNEG0;  % average stiochoimetry
  gradtAvgNEG = 0;          % average gradient of stiochiometry
  thetaTildeAvgPOS = thetaTildePOS0;
  gradtTildeAvgPOS = 0;
  xvecNEG = [thetaAvgNEG;gradtAvgNEG];  % state variables
  xvecPOS = [thetaTildeAvgPOS;gradtTildeAvgPOS];
  
  % Run simulation.
  vcell = 0*iapp;  % pre-allocate output vector
  vcell(1) = UocpTildePOS(thetaTildePOS0) - UocpNEG(thetaNEG0);
  for k = 2:length(iapp)
    % Generate time-varying state-space matrices for NEG and POS.
    % Use BACKWARD EULER METHOD to discretize the continuous-time system 
    % for improved (but not garenteed) stability!
    AcNEG = [0 0; 0 -30*DsNEG];
    ANEG = diag(1./diag(eye(2)-ts*AcNEG)); % inverse of diagonal matrix
    BNEG = -ts*ThQNEG*ANEG*[3; 45/2];
    DNEG = -ThQNEG/(35*DsNEG); 
    AcPOS = [0 0; 0 -30*DsPOS];
    APOS = diag(1./diag(eye(2)-ts*AcPOS)); % inverse of diagonal matrix
    BPOS = -ts*ThQPOS*APOS*[3; 45/2];
    DPOS = -ThQPOS/(35*DsPOS); 

    % NEG update ----------------------------------------------------------
    % Get output (surface stiochiometry) for this iteration.
    ifNEG = iapp(k);
    thetassNEG = CNEG*xvecNEG + DNEG*ifNEG;
    thetassNEG = min(thetassNEG,0.999);
    thetassNEG = max(thetassNEG,0.001);

    % Compute stiochiometry at radius rTildeRef inside particle.
    % Clamp at theta0NEG.
    thetaRefNEG = -35*thetaAvgNEG*(1/4 -rTildeRef^2 +(3/4)*rTildeRef^4)...
        + thetassNEG*(39/4 - 35*rTildeRef^2 + (105/4)*rTildeRef^4)...
        - gradtAvgNEG*(3 - 10*rTildeRef^2 + 7*rTildeRef^4);
    if thetaRefNEG < theta0NEG
        thetaRefNEG = theta0NEG;
    end
    
    % Propagate state equation.
    xvecNEG = ANEG*xvecNEG + BNEG*ifNEG;
    thetaAvgNEG = xvecNEG(1);
    gradtAvgNEG = xvecNEG(2);
    
    % Update Ds.
    DsNEG = DsrefNEG*(F/(R*T))*thetaRefNEG*...
            (thetaRefNEG-1)*dUocpNEG(thetaRefNEG);
    
    % POS update ----------------------------------------------------------
    % Get output (relative surface stiochiometry) for this iteration.
    ifPOS = -iapp(k);
    thetassTildePOS = CPOS*xvecPOS + DPOS*ifPOS;
    thetassTildePOS = min(thetassTildePOS,0.999);
    thetassTildePOS = max(thetassTildePOS,0.001);
    
    % Compute relative stiochiometry at radius rTildeRef inside particle.
    thetaTildeRefPOS = -35*thetaTildeAvgPOS*(1/4 -rTildeRef^2 +(3/4)*rTildeRef^4)...
        + thetassTildePOS*(39/4 - 35*rTildeRef^2 + (105/4)*rTildeRef^4)...
        - gradtTildeAvgPOS*(3 - 10*rTildeRef^2 + 7*rTildeRef^4);
    
    % Propagate state equation.
    xvecPOS = APOS*xvecPOS + BPOS*ifPOS;
    thetaTildeAvgPOS = xvecPOS(1);
    gradtTildeAvgPOS = xvecPOS(2);
    
    % Update Ds.
    DsPOS = abs(DsrefPOS*(F/(R*T))*(1-(m/b)*(b-thetaTildeRefPOS))*...
        (thetaTildeRefPOS-b)*dUocpTildePOS(thetaTildeRefPOS));

    % Cell voltage update -------------------------------------------------
    vcell(k) = UocpTildePOS(thetassTildePOS)-UocpNEG(thetassNEG);
  end % for

end % runSPM()

% STUFF Stuff a structure of optimization parameters into a vector.
function vector = stuff(structure,pinned)
  paramnames = {'DsrefNEG','DsrefPOS','mPOS','bPOS'};
  nparam = length(paramnames);
  npin = length(fieldnames(pinned));
  vector = zeros(nparam-npin,1);
  cursor = 1;
  for k = 1:nparam
    pname = paramnames{k};
    if ~isfield(pinned,pname)
      if any(strcmp(pname,{'DsrefNEG','DsrefPOS'}))
        % Use log scale for diffusion coefficients.
        vector(cursor) = log10(structure.(pname));
      else
        vector(cursor) = structure.(pname);
      end
      cursor = cursor + 1;
    end
  end % for
end % stuff()

% EXPLODE Unstuff a vector of optimization parameters into a structure.
function structure = explode(vector,pinned)
  paramnames = {'DsrefNEG','DsrefPOS','mPOS','bPOS'};
  nparam = length(paramnames);
  structure = struct;
  cursor = 1;
  for k = 1:nparam
    pname = paramnames{k};
    if isfield(pinned,pname)
      structure.(pname) = pinned.(pname);
    else
      if any(strcmp(pname,{'DsrefNEG','DsrefPOS'}))
        % Use log scale for diffusion coefficients.
        structure.(pname) = 10^(vector(cursor));
      else
        structure.(pname) = vector(cursor);
      end
      cursor = cursor + 1;
    end
  end % for
end % explode()

% EVALSTRINGS Evalulate any strings in a nested struct using eval().
% !!! DO NOT REPLACE `params` WITH `~`, as call to eval() may use it!
function s = evalstrings(s,params)  %#ok<INUSD> 
  secnames = fieldnames(s);
  for ks = 1:length(secnames)
    sname = secnames{ks};
    sec = s.(sname);
    if isstruct(sec)
      paramnames = fieldnames(sec);
      for kp = 1:length(paramnames)
        pname = paramnames{kp};
        value = sec.(pname);
        if ischar(value) || isstring(value)
          s.(sname).(pname) = eval(value);
        end
      end % for
    else
      value = sec;
      if ischar(value) || isstring(value)
        s.(sname) = eval(value);
      end
    end
  end % for
end % function