%--------------------------------------------------------------------------
% This function will compute (generalized) DRT from given impedance data
%
% Inputs:  freq (data frequency)
%          Zdata (impedance data, complex numbers)
%          lambda (regularization parameter)
%          coeff (discretization parameter)
% Outputs: Zdrt (reconstruct impedance)
%          tau (time constants)
%          gamma (DRT function)
%          R (series resistance)
%          L (series inductance)
%          C (series capacitor)
%
% The DRT code for series RC follows the work from:
% [1] T.H. Wan, M. Saccoccio, C. Chen, F. Ciucci, Influence of the
%     Discretization Methods on the Distribution of Relaxation Times
%     Deconvolution: Implementing Radial Basis Functions with DRTtools
%     Website: https://ciucci.org/project/drt/
%     Github:  https://github.com/ciuccislab
% [2] M. Saccoccio, T.H. Wan, C. Chen, and F. Ciucci. Optimal
%     Regularization in Distribution of Relaxation Times Applied to
%     Electrochemical Impedance Spectroscopy: Ridge and Lasso Regression
%     Methods– a Theoretical and Experimental Study
% [3] F. Ciucci, C. Chen, Analysis of Electrochemical Impedance
%     Spectroscopy Data Using the Distribution of Relaxation Times: A
%     Bayesian and Hierarchical Bayesian Approach
% [4] M.B. Effat, F. Ciucci, Bayesian and Hierarchical Bayesian Based
%     Regularization for Deconvolving the Distribution of Relaxation
%     Times from Electrochemical Impedance Spectroscopy Data
%
% For generalized DRT:
% [5] Danzer MA. Generalized Distribution of Relaxation Times Analysis for 
%     the Characterization of Impedance Spectra. Batteries. 
%--------------------------------------------------------------------------
function data = computeGDRT(freq,Zdata,lambda,coeff)

  %% Extract input data
  %*********************
  freq  = freq(:);
  Zdata = Zdata(:); 
  
  % Sort frequencies from high to low
  [freq,ind] = sort(freq,'descend'); 
  Zdata = Zdata(ind);
  s = 1j*2*pi*freq;
  
  % See if user inputs "lambda" and "coeff" values
  % lambda: regularization parameter
  % coeff: FWHM shape control<=1 (larger values make DRT oscillatory)
  if nargin==2,lambda = 1e-11; coeff = 0.5;end
  if nargin==3,coeff  = 0.5;end
  
  % Other DRT parameters that will not be changed 
  deriv = '1st-order'; % derivative penalties ('1st-' or '2nd-order')
  
  %% Discretizations (radial basis function, RBFs)
  %************************************************
  % Setup Gaussian radial basis function (RBF)
  % FWHM = FWHMcoeff/epsilon
  FWHM = @(x) exp(-(x).^2)-1/2; % mu = 1
  FWHMcoeff = 2*fzero(@(x) FWHM(x),1); % FWHM
  
  % Calculate the scale-factor (epsilon)
  % In ref [1], they use "mu" instead of "epsilon"
  % epsilon = m*FWHMcoeff/delta(ln(tau))
  delta   = mean(diff(log(1./freq))); % ln(tau)-ln(taum)
  epsilon = coeff*FWHMcoeff/delta;
  
  %% (Tikhonov) regularization 
  %*****************************
  % Other than the two DRT functions, we also identify R, L and C
  % Reserve the first three entries for them
  indR  = 1;
  indL  = 2;
  indC  = 3;
  
  Nf = numel(freq); % number of frequencies
  Nc = indC; % number of costant elements 
  indRC = Nc+(1:Nf);
  indRL = Nc+(Nf+1:2*Nf);
  
  %---------------------------------------
  % Build some matrices
  %---------------------------------------
  % Build A matrices: Eq.(32-33), ref [1]
  % Compute R//C and R//L separately
  [AreRC,AimRC] = assembleA(freq,epsilon,'RC');
  [AreRL,AimRL] = assembleA(freq,epsilon,'RL');
  
  Are = [zeros(Nf,Nc) AreRC AreRL];
  Aim = [zeros(Nf,Nc) AimRC AimRL];
  
  % Adding the multipliers for R, L, and 1/C
  Are(:,1) = 1; % Z = [1]*R
  Aim(:,2) = 2*pi*freq; % Z = j*[2*pi*f]*L
  Aim(:,3) = -1./(2*pi*freq); % Z = j*[-1/(2*pi*f)]/C
  
  % Build M matrix: Eq.(38), ref [1]
  M  = zeros(Nc+Nf*2,Nc+Nf*2);
  mm = assembleM(freq,epsilon,deriv);
  M(indRC,indRC) = mm;
  M(indRL,indRL) = mm;
  
  % Build b (data) matrices
  bre = real(Zdata);
  bim = imag(Zdata);
  
  %-------------------------------------------------------
  % Setup the initial values and boundary conditions
  %-------------------------------------------------------
  % Setup initial and boundaries for regression
  % Remember the first three entries are reserved for R, L, and C
  % x = [R;L;1/C;xRC;xRL]
  x0 = ones(Nc+Nf*2,1)*1e-3; 
  lb = ones(Nc+Nf*2,1)*0;    
  ub = ones(Nc+Nf*2,1)*inf;  
  
  % Convert the cost function in MATLAB quadratic programming format
  HH = 2*((Are'*Are+Aim'*Aim)+lambda*M);
  HH = (HH'+HH)/2;
  ff = -2*(Aim'*bim+Are'*bre);
  
  % % options = optimset('algorithm','active-set','display','off',...
  % %     'TolFun',1e-15,'TolX',1e-15,'TolCon',1e-15,'MaxFunEvals',1e6,'MaxIter',1e6);
  options = optimset('algorithm','active-set','display','off',...
      'TolFun',1e-15,'TolX',1e-15,'TolCon',1e-15,'MaxFunEvals',1e6,'MaxIter',1e6);
  warning('off','MATLAB:singularMatrix');
  
  %----------------------------------
  % Run optimization
  %----------------------------------
  % It was found that the results depend on the lower boundary of R. However,
  % we also know that the frequencies for RL-DRT must be higher than the
  % RC-DRT. The following scripts will try different R values until their
  % frequencies meet the criteria
  xEstStore = [];
  ZreMin    = abs(min(bre));
  for k = 1:-0.1:0.5
    x0(indR) = ZreMin*k;
    lb(indR) = x0(indR); % set R as constant
    ub(indR) = x0(indR); % set R as constant
    xEst = quadprog(HH,ff,[],[],[],[],lb,ub,x0,options);
    xEst(xEst<1e-8) = 0;
    xEstStore = [xEstStore xEst]; %#ok<AGROW>
    
    [~,ind1] = findpeaks(xEst(indRC)); if xEst(indRC(1))>0,ind1 = [1;ind1];end %#ok<AGROW>
    [~,ind2] = findpeaks(xEst(indRL)); if xEst(indRL(1))>0,ind2 = [1;ind2];end %#ok<AGROW>
    
    % Enter the following logic means there is change in results
    % This setup help find results faster, rather than just k=1:-0.01:0.5
    if or(ind1(1)<ind2(end),ind1(end)<ind2(end))
      if k==1,xEst = xEstStore;else,xEst = xEstStore(:,end-1);end
      break;
    end
  end
  
  xEstStore = [];
  for p = k+0.1:-0.01:k
    if p>1,break;end
    x0(indR) = ZreMin*p;
    lb(indR) = x0(indR); % set R as constant
    ub(indR) = x0(indR); % set R as constant
    xEst = quadprog(HH,ff,[],[],[],[],lb,ub,x0,options);
    xEst(xEst<1e-8) = 0;
    xEstStore = [xEstStore xEst]; %#ok<AGROW>
    
    [~,ind1] = findpeaks(xEst(indRC)); if xEst(indRC(1))>0,ind1 = [1;ind1];end %#ok<AGROW>
    [~,ind2] = findpeaks(xEst(indRL)); if xEst(indRL(1))>0,ind2 = [1;ind2];end %#ok<AGROW>
    if or(ind1(1)<ind2(end),ind1(end)<ind2(end))
      if p==k+0.1,xEst = xEstStore;else,xEst = xEstStore(:,end-1);end
      break;
    end
  end
  
  %-----------------------------
  % Extract results
  %-----------------------------
  R = xEst(indR);
  L = xEst(indL);
  C = 1/xEst(indC);
  xmRC = xEst(Nc+1:Nc+Nf);
  xmRL = xEst(Nc+Nf+1:end);
  
  %% Output results
  %*****************
  % Reconstruct impedance using the estimated DRT magnitudes
  Zdrt    = Are*xEst + 1j*Aim*xEst; % overall impedance: R+sL+1/(sC0)+Z_RC+Z_RL
  Zdrt_RC = AreRC*xmRC + 1j*AimRC*xmRC + R + 1./(s*C); 
  Zdrt_RL = AreRL*xmRL + 1j*AimRL*xmRL + R + s*L; 
  
  % Compute the DRT function
  tauMax  = ceil(max(log10(1./freq)))+1;  % round ln(tau) towards +inf
  tauMin  = floor(min(log10(1./freq)))-1; % round ln(tau) towards -inf
  freqTau = logspace(-tauMin,-tauMax,10*Nf); % 10-times more than f
  [gamma_RC,gamma_RL] = mapArrayToGamma(freqTau,freq,xEst(Nc+1:end),epsilon);
  
  % Function output
  data = [];
  data.freq  = freq;
  data.Zdata = Zdata;
  data.lambda= lambda; 
  data.coeff = coeff;
  
  data.R = R;
  data.L = L;
  data.C = C; 
  data.tau      = 1./freqTau(:); % time-constant tau
  data.gamma    = gamma_RC+gamma_RL; % distribution functions
  data.gamma_RC = gamma_RC; 
  data.gamma_RL = gamma_RL;
  data.Zdrt    = Zdrt; % reconstructed impedance 
  data.Zdrt_RC = Zdrt_RC;
  data.Zdrt_RL = Zdrt_RL;
end
  
%% This function assembles the A matrix
%****************************************
function [outAre,outAim] = assembleA(freq,epsilon,type)
  % We will assume that "tau" vector is identical to "freq" vector
  Nf  = numel(freq);
  rbf = @(x) exp(-(epsilon*x).^2); % default "Gaussian" RBF
  
  % Create empty storage
  outAre_temp = zeros(Nf);
  outAim_temp = zeros(Nf);
  
  % We compute if the frequencies are sufficiently log spaced
  % If they are, we apply the toeplitz trick
  std_diff_freq  = std(diff(log(1./freq)));
  mean_diff_freq = mean(diff(log(1./freq)));
  toeplitz_trick = std_diff_freq/mean_diff_freq<0.01;
  
  % If terms are evenly distributed, we do compute only N terms
  if toeplitz_trick
    Rre = zeros(1,Nf); % first row of the Toeplitz matrix
    Rim = zeros(1,Nf);
    Cre = zeros(Nf,1); % first column of the Toeplitz matrix
    Cim = zeros(Nf,1);
    
    % For clarity the C and R computate separately
    for n = 1:Nf
      freq_n = freq(n);
      freq_m = freq(1);
      [Cre(n,1),Cim(n,1)] = findAnm(freq_n,freq_m);
    end
    
    for m = 1:Nf
      freq_n = freq(1);
      freq_m = freq(m);
      [Rre(1,m),Rim(1,m)] = findAnm(freq_n,freq_m);
    end
    outAre_temp = toeplitz(Cre,Rre);
    outAim_temp = toeplitz(Cim,Rim);
  else
    % Compute rbf with brute force
    for n = 1:Nf
      for m = 1:Nf
        freq_n = freq(n);
        freq_m = freq(m);
        
        % Compute all RBF terms
        [outAre_temp(n,m),outAim_temp(n,m)] = findAnm(freq_n,freq_m);
      end
    end
  end
  
  % Output the functions
  outAre = outAre_temp;
  outAim = outAim_temp;

  %-----------------------------------------------------
  % This function generate the elements of Are and Aim
  % Eq. (32) and (33) in ref. 1
  %-----------------------------------------------------
  function [outAnm_re,outAnm_im] = findAnm(freq_n,freq_m)
    alpha = 2*pi*freq_n/freq_m;
    
    % Integrate the following function from -infty to +infty
    switch type
      case 'RC'
        Anm_re = @(x) 1./(1+alpha^2*exp(2*x)).*rbf(x); % Eq.32
        Anm_im = @(x) alpha./(1./exp(x)+alpha^2*exp(x)).*rbf(x); % Eq.33
        outAnm_re = +integral(Anm_re,-inf,inf,'RelTol',1E-9,'AbsTol',1E-9); % real-part
        outAnm_im = -integral(Anm_im,-inf,inf,'RelTol',1E-9,'AbsTol',1E-9); % imag-part
      case 'RL'
        Anm_re = @(x) alpha^2*exp(x)./(1+alpha^2*exp(2*x)).*rbf(x); 
        Anm_im = @(x) alpha./(1+alpha^2*exp(2*x)).*rbf(x); 
        outAnm_re = +integral(Anm_re,-inf,inf,'RelTol',1E-9,'AbsTol',1E-9);
        outAnm_im = +integral(Anm_im,-inf,inf,'RelTol',1E-9,'AbsTol',1E-9);
    end
  end
end

%% This function assembles the M matrix
%*****************************************
function outM = assembleM(freq,epsilon,deriv)
  % The M matrix contains the inner products of 1st-derivative of the discretization RBF.
  % Size of M matrix depends on the number of collocation points, i.e. "tau" vector
  % We assume that the "tau" vector is the inverse of the "freq" vector
  Nf = numel(freq);
  
  % Create empty storage
  outM_temp = zeros(Nf);
  
  % Compute if the frequencies are sufficienly log spaced
  % If they are, then we apply the toeplitz trick
  std_diff_freq  = std(diff(log(1./freq)));
  mean_diff_freq = mean(diff(log(1./freq)));
  toeplitz_trick = std_diff_freq/mean_diff_freq<0.01;
  
  % If terms are evenly distributed, then we do compute only N terms
  if toeplitz_trick
    R = zeros(1,Nf);
    C = zeros(Nf,1);
    
    % For clarity the C and R computations are separated
    for n = 1:Nf
      freq_n = freq(n);
      freq_m = freq(1);
      C(n,1) = inner_prod_rbf(freq_n,freq_m,epsilon,deriv);
    end
    
    for m = 1:Nf
      freq_n = freq(1);
      freq_m = freq(m);
      R(1,m) = inner_prod_rbf(freq_n,freq_m,epsilon,deriv);
    end
    outM_temp = toeplitz(C,R);           
  else
    % Compute rbf with brute force
    for n = 1:Nf
      for m = 1:Nf
        freq_n = freq(n);
        freq_m = freq(m);
        outM_temp(n,m) = inner_prod_rbf(freq_n,freq_m,epsilon,deriv);
      end
    end
    % out_L_temp = chol(outM_temp);
  end
  
  % Output the functions
  outM = outM_temp;

  %----------------------------------------------------------------------
  % This function output the inner product of the first or second
  % derivative of the rbf with respect to log(1/freq_n) and
  % log(1/freq_m). Eq. (38) in ref. 1
  %----------------------------------------------------------------------
  function outIP = inner_prod_rbf(freq_n,freq_m,epsilon,deriv)
    a = epsilon*log(freq_n/freq_m);
    
    % Default "Gaussian" RBF
    switch deriv
      case '1st-order',outIP = -epsilon*(-1+a^2)*exp(-(a^2/2))*sqrt(pi/2);
      case '2nd-order',outIP = epsilon^3*(3-6*a^2+a^4)*exp(-(a^2/2))*sqrt(pi/2);
      otherwise,error('Incorrect derivative input!');
    end
  end
end
  
%% This function performs the mappings, Eq.(29), ref 2
%********************************************************
function [outGamma_RC,outGamma_RL] = mapArrayToGamma(freqMap,freqData,x,epsilon)
  % This function map "x" (i.e., the magnitude at the specific frequency)
  % to "gamma" (i.e., the DRT profile).
  % Multiplying rbf function with x, one can obtain the gamma profile.
  Nf = numel(freqData);
  
  % "Gaussian" RBF is default
  rbf = @(y,y0) exp(-(epsilon*(y-y0)).^2);
  y0  = -log(freqData);
  
  % Create empty storage
  outGamma_RC = zeros(numel(freqMap),1);
  outGamma_RL = zeros(numel(freqMap),1);
  
  % Loop over all "tau"
  for i = 1:numel(freqMap)
    y = -log(freqMap(i));
    
    outGamma_RC(i) = x(1:Nf)'*rbf(y,y0);
    if Nf<numel(x),outGamma_RL(i) = x(Nf+1:end)'*rbf(y,y0);end
  end
end