% CALIBRATESOC Convert OCP-v-time data to OCP-v-SOC data.
%
% -- Usage --
% calData = CALIBRATESOC(rawData,dv)
%
% -- Input --
% rawData    : struct vector with the following fields:
%   .TdegC   : test temperature [degC]
%   .script1 : voltage-v-time during script 1 (from makeMATfile[OCP|OCV])
%   .script2 : voltage-v-time during script 2 (from makeMATfile[OCP|OCV])
%   .script3 : voltage-v-time during script 3 (from makeMATfile[OCP|OCV])
%   .script4 : voltage-v-time during script 4 (from makeMATfile[OCP|OCV])
%
% -- Output --
% calData    : struct vector with the following fields:
%   .TdegC   : test temperature [degC]
%   .C_rate_actual : average C-rate during dis/charge steps [C-rate]
%   .QAh     : estimated capacity [Ah]
%   .eta     : estimated coloumbic efficiency [-]
%   .disZ    : electrode SOC (relative composition) vector for discharge [-]
%   .disV    : voltage vector for discharge [V]
%   .disdZ   : differential capacity vector for discharge [1/V]
%   .chgZ    : electorde SOC (relative composition) vector for charge [-]
%   .chgV    : voltage vector for charge [V]
%   .chgdZ   : differential capacity vector for charge [1/V]
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function calData = calibrateSOC(rawData,dv)

  if ~exist('dv','var')
    dv = 1e-3;
  end
  
  % Define some quantities
  calData(length(rawData)) = struct( ...
    'TdegC',[],'QAh',[],'eta',[], ...
    'disZ',[],'disV',[],'disdZ',[],'chgZ',[],'chgV',[],'chgdZ',[]);
  vectTdegC = [rawData.TdegC]';
  
  ind25 = find(vectTdegC == 25);
  not25 = find(vectTdegC ~= 25);
  if isempty(ind25)
    error('Must have a reference test at 25degC. Cannot continue.');
  end
  
  %--------------------------------------
  % Process data at 25C
  %--------------------------------------
  % NOTE: Generally, there is only one test at 25degC, but in the case that
  % there are multiple tests at 25degC, we process them all and use the last
  % test as the reference for non-25degC tests.
  % An exception is the discharge test, in which all tests are performed at
  % 25degC but at different C-rates.
  for k = ind25'
    totDisAh25 = sum(arrayfun(@(nbr)rawData(k).(sprintf('script%d',nbr)).disAh(end),1:4));
    totChgAh25 = sum(arrayfun(@(nbr)rawData(k).(sprintf('script%d',nbr)).chgAh(end),1:4));
    s12DisAh25 = sum(arrayfun(@(nbr)rawData(k).(sprintf('script%d',nbr)).disAh(end),1:2));
    s12ChgAh25 = sum(arrayfun(@(nbr)rawData(k).(sprintf('script%d',nbr)).chgAh(end),1:2));
    
    % Compute coulombic efficiency and capacity
    eta25 = totDisAh25/totChgAh25;  % 25 degC coulombic efficiency
    QAh25 = s12DisAh25 - eta25*s12ChgAh25;
    
    % Compute dis/charge data SOC
    indDis = find(rawData(k).script1.step == 2);
    indChg = find(rawData(k).script3.step == 2);
    s1DisAh = rawData(k).script1.disAh(indDis);
    s1ChgAh = rawData(k).script1.chgAh(indDis);
    s3DisAh = rawData(k).script3.disAh(indChg);
    s3ChgAh = rawData(k).script3.chgAh(indChg);
    disZ = (s1DisAh - eta25*s1ChgAh)/QAh25;
    chgZ = (s3DisAh - eta25*s3ChgAh)/QAh25;
    disVt = rawData(k).script1.voltage(indDis);  % vs time (t)
    chgVt = rawData(k).script3.voltage(indChg);  % vs time (t)
    disIt = rawData(k).script1.current(indDis);  % vs time (t)
    chgIt = rawData(k).script3.current(indChg);  % vs time (t)
    disT = rawData(k).script1.time(indDis);
    chgT = rawData(k).script3.time(indChg);
    if mean(disIt)<0
      disIt = -disIt; % correct the sign convention!
    end
    if mean(chgIt)>0
      chgIt = -chgIt; % correct the sign convention!
    end
    
    % Due to floating point arithmetic error in numerical integration, 
    % sometimes disZ does not start precisely at 0 and chgZ does not 
    % start precisely at 1; subtract off the differences.
    disZ = disZ - disZ(1);
    chgZ = chgZ - chgZ(1) + 1;
    
    % Smooth voltage-v-soc curves and compute differential capacity.
    [disZ,disV,disdZ] = smoothdiff(disZ,disVt,dv); % disV now vs SOC
    [chgZ,chgV,chgdZ] = smoothdiff(chgZ,chgVt,dv); % chgV now vs SOC
    
    % Ensure V is monotonicly increasing (not decreasing)
    if disV(1)>disV(end)
      disZ  = flip(disZ);
      disV  = flip(disV);
      disdZ = flip(disdZ);
    end
    if chgV(1)>chgV(end)
      chgZ  = flip(chgZ);
      chgV  = flip(chgV);
      chgdZ = flip(chgdZ);
    end
    
    % Check that SOC remains between 0% and 100%.
    % Numerical error could 
    if max(disZ) > 1
      outp.warning( ...
          "SOC > 100%% [min:%.3f%%] on discharge for '%s' @ T=%.2fdegC\n", ...
          max(disZ)*100, rawData(k).name, rawData(k).TdegC);
    end
    if min(chgZ) < 0
      outp.warning( ...
          "SOC < 0%% [min:%.3f%%] on charge for '%s' @ T=%.2fdegC\n", ...
          min(chgZ)*100, rawData(k).name, rawData(k).TdegC);
    end
    
    % Store processed data.
    calData(k).TdegC = rawData(k).TdegC;
    calData(k).C_rate_actual = mean(abs([disIt(:);chgIt(:)]))/QAh25;
    calData(k).eta = eta25;
    calData(k).QAh = QAh25;
    calData(k).disZ = disZ;
    calData(k).disV = disV;
    calData(k).disdZ = disdZ;
    calData(k).chgZ = chgZ;
    calData(k).chgV = chgV;
    calData(k).chgdZ = chgdZ;
    calData(k).disT = disT-disT(1); % start from t=0
    calData(k).disIt = disIt;
    calData(k).disVt = disVt;
    calData(k).chgT = chgT-chgT(1); % start from t=0
    calData(k).chgIt = chgIt;
    calData(k).chgVt = chgVt;
    
    % Adjust charge Ah in all scripts
    % rawData(k).script1.chgAh = rawData(k).script1.chgAh*eta25;
    % rawData(k).script2.chgAh = rawData(k).script2.chgAh*eta25;
    % rawData(k).script3.chgAh = rawData(k).script3.chgAh*eta25;
    % rawData(k).script4.chgAh = rawData(k).script4.chgAh*eta25;
  end
  
  %--------------------------------------------------
  % Process data at other degree C
  %--------------------------------------------------
  for k = not25'
    totDisAh = sum(arrayfun(@(nbr)rawData(k).(sprintf('script%d',nbr)).disAh(end),1:4));
    s12DisAh = sum(arrayfun(@(nbr)rawData(k).(sprintf('script%d',nbr)).disAh(end),1:2));
    s1ChgAh = sum(arrayfun(@(nbr)rawData(k).(sprintf('script%d',nbr)).chgAh(end),1)); % charged @ TdegC 
    s2ChgAh = sum(arrayfun(@(nbr)rawData(k).(sprintf('script%d',nbr)).chgAh(end),2)); % charged @ 25degC
    s13ChgAh = sum(arrayfun(@(nbr)rawData(k).(sprintf('script%d',nbr)).chgAh(end),[1,3])); % charged @ TdegC
    s24ChgAh = sum(arrayfun(@(nbr)rawData(k).(sprintf('script%d',nbr)).chgAh(end),[2,4])); % charged @ 25degC
    
    % Compute coulombic efficiency and capacity
    eta = (totDisAh - eta25*s24ChgAh)/s13ChgAh;
    QAh = s12DisAh - eta*s1ChgAh - eta25*s2ChgAh;
    
    % Compute dis/charge data SOC
    indDis = find(rawData(k).script1.step == 2);
    indChg = find(rawData(k).script3.step == 2);
    s1DisAh = rawData(k).script1.disAh(indDis);
    s1ChgAh = rawData(k).script1.chgAh(indDis);
    s3DisAh = rawData(k).script3.disAh(indChg);
    s3ChgAh = rawData(k).script3.chgAh(indChg);
    disZ = (s1DisAh - eta*s1ChgAh)/QAh;
    chgZ = (s3DisAh - eta*s3ChgAh)/QAh;
    disVt = rawData(k).script1.voltage(indDis);  % vs time (t)
    chgVt = rawData(k).script3.voltage(indChg);  % vs time (t)
    disIt = rawData(k).script1.current(indDis);
    chgIt = rawData(k).script3.current(indChg);
    disT = rawData(k).script1.time(indDis);
    chgT = rawData(k).script3.time(indChg);
    if mean(disIt)<0
        disIt = -disIt; % correct the sign convention!
    end
    if mean(chgIt)>0
        chgIt = -chgIt; % correct the sign convention!
    end
    
    % Due to floating point arithmetic error in numerical integration, 
    % sometimes disZ does not start precisely at 0 and chgZ does not 
    % start precisely at 1; subtract off the differences.
    disZ = disZ - disZ(1);
    chgZ = chgZ - chgZ(1) + 1;
    
    % Smooth voltage-v-soc curves and compute differential capacity.
    [disZ,disV,disdZ] = smoothdiff(disZ,disVt,dv); % disV now vs SOC
    [chgZ,chgV,chgdZ] = smoothdiff(chgZ,chgVt,dv); % chgV now vs SOC
    
    % Ensure V is monotonicly increasing (not decreasing)
    if disV(1)>disV(end)
      disZ  = flip(disZ);
      disV  = flip(disV);
      disdZ = flip(disdZ);
    end
    if chgV(1)>chgV(end)
      chgZ  = flip(chgZ);
      chgV  = flip(chgV);
      chgdZ = flip(chgdZ);
    end
    
    % Check that SOC remains between 0% and 100%.
    if max(disZ) > 1
      outp.warning( ...
          "SOC > 100%% [min:%.3f%%] on discharge for '%s' @ T=%.2fdegC\n", ...
          max(disZ)*100, rawData(k).name, rawData(k).TdegC);
    end
    if min(chgZ) < 0
      outp.warning( ...
          "SOC < 0%% [min:%.3f%%] on charge for '%s' @ T=%.2fdegC\n", ...
          min(chgZ)*100, rawData(k).name, rawData(k).TdegC);
    end
    
    % Store processed data.
    calData(k).TdegC = rawData(k).TdegC;
    calData(k).C_rate_actual = mean(abs([disIt(:);chgIt(:)]))/QAh;
    calData(k).eta = eta;
    calData(k).QAh = QAh;
    calData(k).disZ = disZ;
    calData(k).disV = disV;
    calData(k).disdZ = disdZ;
    calData(k).chgZ = chgZ;
    calData(k).chgV = chgV;
    calData(k).chgdZ = chgdZ;
    calData(k).disT = disT-disT(1); % start from t=0
    calData(k).disIt = disIt;
    calData(k).disVt = disVt;
    calData(k).chgT = chgT-chgT(1); % start from t=0
    calData(k).chgIt = chgIt;
    calData(k).chgVt = chgVt;
    
    % Adjust charge Ah in all scripts
    % rawData(k).script1.chgAh = rawData(k).script1.chgAh*eta;
    % rawData(k).script2.chgAh = rawData(k).script2.chgAh*eta25;
    % rawData(k).script3.chgAh = rawData(k).script3.chgAh*eta;
    % rawData(k).script4.chgAh = rawData(k).script4.chgAh*eta25;
  end
end