%UNPACK Unstuff a model structure from a vector.
%
% model = UNPACK(vect, metadata) converts the vector VECT back into
%   a model structure using the metadata METADATA.
%
% model = UNPACK(...,'sparse') omits fixed parameter values from
%   the model.
%
% model = UNPACK(...,'flat') returns a flat-parameter model
%   instead of a hierarchical model.
%
% model = UNPACK(...,'fixed') returns only the values of fixed
%   parameters. In this case, the input VECT is optional and may be
%   replaced by the empty vector [].
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function model = unpack(vect, metadata, varargin)

  sparse = any(strcmp(varargin,'sparse'));
  flat = any(strcmp(varargin,'flat'));
  fixedOnly = any(strcmp(varargin,'fixed'));
  notemp = any(strcmp(varargin,'notemp'));

  flatmodel = struct;
  vect = vect(:);

  % Extract parameters.
  paramnames = fieldnames(metadata.params);
  cursor = 1;
  for k = 1:length(paramnames)
    paramname = paramnames{k};
    meta = metadata.params.(paramname);

    % Determine multiplicity.
    mult = 1;
    if strcmpi(meta.tempfcn,'lut')
      mult = metadata.ntemps;
    end

    % Determine length.
    len = meta.len;
    if isfield(meta,'fix')
      len = sum(meta.fixmask);
    end

    % Extract value and store into model structure.
    if isempty(vect)
      value = [];
    else
      value = vect(cursor:cursor+len*mult-1);
      value = reshape(value,[len,mult]);
    end
    if isfield(meta,'fix')
      if ~sparse
        flatmodel.(paramname) = meta.fix;
        flatmodel.(paramname)(meta.fixmask) = value;
      elseif len > 0
        % Some components are not fixed; omit the fixed components
        % and retain the unfixed components.
        flatmodel.(paramname) = value;
      else
        % Fixed parameter; omit from sparse output.
      end
    elseif ~fixedOnly
        % Unfixed parameter.
        flatmodel.(paramname) = value;
    end % if
    cursor = cursor + len*mult;

    % Decode logarithmic values.
    if isfield(meta,'logscale') && meta.logscale == true && isfield(flatmodel,paramname)
      flatmodel.(paramname) = 10.^(flatmodel.(paramname));
    end

    % Check for activation energy.
    if ~notemp && strcmpi(meta.tempfcn,'Eact') && ~isempty(vect)
      % Activation energy is present; extract from vector.
      paramnameEact = [paramname '_Eact'];
      Eact = vect(cursor:cursor+len-1);
      flatmodel.(paramnameEact) = Eact;
      cursor = cursor + len;
    end % if
  end % for

  if flat
    model = flatmodel;
  else
    % Deflatten model.
    model = struct;
    paramnames = fieldnames(flatmodel);
    for k = 1:length(paramnames)
      paramname = paramnames{k};
      parts = split(paramname,'__');
      model = setfield(model,parts{:},flatmodel.(paramname));
    end
  end
end