% UIPARTICLESWARM Construct a GUI for running PSO interactively.
%
% -- Usage --
% data = uiparticleswarm(costFcn,modelspec,init,lb,ub)
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function guidata = uiparticleswarm(costFcn,modelspec,init,lb,ub,varargin)
  
  parser = inputParser;
  parser.addRequired('costFcn',@(x)isa(x,'function_handle'));
  parser.addRequired('modelspec',@isstruct);
  parser.addRequired('init',@isstruct);
  parser.addRequired('lb',@isstruct);
  parser.addRequired('ub',@isstruct);
  parser.addParameter('SwarmSize',500);
  parser.addParameter('SwarmMaximumIterations',100);
  parser.addParameter('FminMaximumIterations',1000);
  parser.addParameter('FminMaximumFunctionEvals',10000);
  parser.addParameter('PlotInitializeFcn', ...
      @(varargin)[],@(x)isa(x,'function_handle'));
  parser.addParameter('PlotUpdateFcn', ...
      @(varargin)[],@(x)isa(x,'function_handle'));
  parser.addParameter('WindowName','Particle Swarm Optimization');
  parser.addParameter('UseParallel',false,@(x)islogical(x)&&isscalar(x));
  parser.addParameter('UtilOpts',{},@iscell);
  parser.parse(costFcn,modelspec,init,lb,ub,varargin{:});
  arg = parser.Results;  % structure of validated arguments
  
  % Ensure initial values are between lb and ub.
  % We also pack/unpack the lb and ub structures to apply coerce.
  lb = optutil.pack(arg.lb,arg.modelspec,[],'coerce',arg.UtilOpts{:});
  ub = optutil.pack(arg.ub,arg.modelspec,[],'coerce',arg.UtilOpts{:});
  init = optutil.pack(arg.init,arg.modelspec,[],'coerce',arg.UtilOpts{:});
  init(init<lb) = lb(init<lb);
  init(init>ub) = ub(init>ub);
  arg.lb = optutil.unpack(lb,arg.modelspec,arg.UtilOpts{:});
  arg.ub = optutil.unpack(ub,arg.modelspec,arg.UtilOpts{:});
  arg.init = optutil.unpack(init,arg.modelspec,arg.UtilOpts{:});
  
  costFcn = costFunctionFactory(arg.costFcn,arg.modelspec,arg);
  modelspec = arg.modelspec;
  
  guiarg = rmfield(arg,{'modelspec','init','lb','ub','UseParallel'});
  gui = optutil.PSOApp(modelspec,arg.init,arg.lb,arg.ub,guiarg);
  while true
    % Fetch optimization parameters from user in GUI.
    guidata = gui.run();
    lb = optutil.pack(guidata.lb,modelspec,[],arg.UtilOpts{:});
    ub = optutil.pack(guidata.ub,modelspec,[],arg.UtilOpts{:});
    init = optutil.pack(guidata.init,modelspec,[],arg.UtilOpts{:});
    fix = optutil.pack(guidata.fix,modelspec,[],'nolog',arg.UtilOpts{:});

    % Make parameters fixed (the lazy way).
    fix = logical(fix);
    lb(fix) = init(fix);
    ub(fix) = init(fix);

    estimate = init;
    opttype = guidata.OptimizationType;
    popsize = guidata.SwarmSize;
    maxiter = guidata.SwarmMaximumIterations;
    fminmaxiter = guidata.FminMaximumIterations;
    fminfevals = guidata.FminMaximumFunctionEvals;
    if guidata.stop
      % User canceled the optimization.
      break;
    end
    [plotFcnPSO, plotFcnFmin, stopFcn] = PSOPlotFcnFactory(arg,guidata);
    gui.lock(stopFcn);  % prevent changes to ui while optimization is running

    if strcmpi(opttype,'particleswarm')
      % Collect PSO options.
      swarm = zeros(popsize,modelspec.nvars);
      swarm(1,:) = init;
      for k = 1:modelspec.nvars
        swarm(2:end,k) = (ub(k)-lb(k))*rand(popsize-1,1)+lb(k);
      end
      options = optimoptions(@particleswarm,...
          'Display','iter', ...
          'UseParallel',arg.UseParallel,...
          'FunctionTolerance',1e-15, ...
          'SwarmSize',popsize,...
          'InitialSwarmMatrix',swarm,...
          'MaxIterations',maxiter, ...
          'MaxStallIterations',200,...
          'FunValCheck','off', ...
          'OutputFcn',plotFcnPSO);  % don't use 'PlotFcn' option, has issues!
  
      % Run PSO.
      estimate = particleswarm(costFcn,modelspec.nvars,lb,ub,options);
    elseif strcmpi(opttype,'fmincon')
      % Collect fmincon options.
      options = optimoptions(@fmincon, ...
          'Display','iter',...
          'UseParallel',arg.UseParallel,...
          'MaxFunEvals',fminfevals, ...
          'MaxIter',fminmaxiter,...
          'TolFun',1e-15, ...
          'TolX',1e-15, ...
          'TolCon',1e-15, ...
          'OutputFcn',plotFcnFmin);

      % Run fmincon.
      estimate = fmincon(costFcn,init,[],[],[],[],lb,ub,[],options);
    end

    % Update GUI with new estimate.
    gui.update(optutil.unpack(estimate,arg.modelspec,arg.UtilOpts{:}));
    gui.unlock();  % allow changes to ui now that PSO finished
  end
  
  % Collect output data.
  guidata = struct;
  guidata.values = optutil.unpack(estimate,modelspec,arg.UtilOpts{:});
  guidata.lb = optutil.unpack(lb,modelspec,arg.UtilOpts{:});
  guidata.ub = optutil.unpack(ub,modelspec,arg.UtilOpts{:});
  guidata.modelspec = modelspec;
  guidata.arg = arg;
  guidata.origin__ = 'optutil.uiparticleswarm';

end

function [plotFcnPSO, plotFcnFmin, stopFcn] = PSOPlotFcnFactory(arg,guidata)
  plotUpdateIntervalSec = 0.5;
  modelspec = arg.modelspec;
  updatePlotFcn = arg.PlotUpdateFcn;
  updateControlsFcn = guidata.updateControlsFcn;
  updateStatusFcn = guidata.updateStatusFcn;
  updateMark = tic;
  bestJ = Inf;
  bestEstimate = [];
  stop = false;
  plotFcnPSO = @PSOPlot;
  plotFcnFmin = @FminPlot;
  stopFcn = @OptimStop;

  function halt = PSOPlot(optimValues,state)
    halt = stop;
    if halt
      return;
    end
    if ~strcmp(state,'iter')
      return;
    end
    if toc(updateMark) > plotUpdateIntervalSec
      if optimValues.bestfval < bestJ
        bestJ = optimValues.bestfval;
        bestEstimate = optutil.unpack(optimValues.bestx,modelspec,arg.UtilOpts{:});
        updatePlotFcn(bestEstimate);
        updateControlsFcn(bestEstimate);
        updateMark = tic;
      end
    end % if
    statusString = sprintf( ...
        "Iteration: %d   min(J): %.3g   Evals: %d", ...
        optimValues.iteration,optimValues.bestfval,optimValues.funccount);
    updateStatusFcn(statusString);
    drawnow;
  end % PSOPlot()

  function halt = FminPlot(x,optimValues,state)
    halt = stop;
    if halt
      return;
    end
    if ~strcmp(state,'iter')
      return;
    end
    if toc(updateMark) > plotUpdateIntervalSec
      if optimValues.fval < bestJ
          bestJ = optimValues.fval;
          bestEstimate = optutil.unpack(x,modelspec,arg.UtilOpts{:});
          updatePlotFcn(bestEstimate);
          updateControlsFcn(bestEstimate);
          updateMark = tic;
      end
    end % if
    statusString = sprintf( ...
        "Iteration: %d   BestJ: %.3g   Evals: %d", ...
        optimValues.iteration,bestJ,optimValues.funccount);
    updateStatusFcn(statusString);
    drawnow;
  end % PSOPlot()

  function OptimStop()
    % Send stop signal.
    stop = true;
  end
end

function wrapped = costFunctionFactory(costFcn,modelspec,arg)
  wrapped = @costFnWrapper;

  function Jcost = costFnWrapper(vect)
    unpacked = optutil.unpack(vect,modelspec,arg.UtilOpts{:});
    Jcost = costFcn(unpacked);
  end
end
