% PARAM Specify a parameter for optimization.
%
% p = PARAM(...) creates a specification for a parameter with
%   fields set by name-value pairs in (...), similar to constructing a
%   structure using the MATLAB built-in `struct` command.
%
% The available options are:
%
% - init=VALUE: initial value for optimization
% - lb=VALUE  : lower bound for optimization
% - ub=VALUE  : upper bound for optimization
% - fix=VALUE: fix the value of the parameter to VALUE 
%     (do not optimize)
% - fixmask=BITMASK: for vector parameters, enables fixing individual
%     elements of the vector. BITMASK is a logical vector of the same
%     size as the value supplied for the `fix` parameter. Elements with
%     false value are fixed to the corresponding element in the
%     fix VALUE vector.
% - len=LENGTH: for vector parameters, sets length as LENGTH 
%     (default to length 1 scalars)
% - logscale={true|false}: Default false. If true, store the parameter
%     internally on a log10 scale (may help optimizers converge).
% - tempfcn={'fix'|'lut'|'Eact'}: Default 'fix'.
%     When 'fix', the parameter is constant with temperature
%     When 'lut', the parameter takes on a different value for each
%       temperature
%     When 'Eact', the parameter follows an Ahhrenius relationship with
%       temperature governed by an additional activation energy (Eact)
%       parameter. You can specify upper and lower bounds for this
%       parameter by adding arguments to the string: 'Eact:{lb}:{ub}'
%       where ub and lb are activation energy limits in kJ.
%       You may also specify an initial value using:
%       'Eact:{init}:{lb}:{ub}'. The defaults are lb=0,
% - tempcoeff={'+'|'-'}. Default '+'. 
%     When '+', use standard Ahhrenius relationship to temperature-
%       correct the parameter (positive temperature coefficient). 
%     When '-', use inverted Ahhrenius relationship to temperature-
%       correct the parameter (negative temperature coefficient).
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function p = param(varargin)

  % Create structure from suppied parameters.
  p = struct(varargin{:});

  paramNames = fieldnames(p);
  paramIsValid = ismember( ...
      paramNames, ...
      {'init','lb','ub','fix','fixmask','len', ...
      'logscale','tempfcn','tempcoeff','EactLB','EactUB','EactINIT'});
  if ~all(paramIsValid)
    invalidParamNames = paramNames(~paramIsValid);
    error('Unrecognized arguments: %s',strjoin(invalidParamNames));
  end

  % Set defaults / enforce integrity.
  if isfield(p,'fix'),         p.len = length(p.fix);         end
  if ~isfield(p,'len'),        p.len = 1;                     end
  if ~isfield(p,'fixmask'),    p.fixmask = false(p.len,1);    end 
  if ~isfield(p,'logscale'),   p.logscale = false;            end
  if ~isfield(p,'tempfcn'),    p.tempfcn = 'fix';             end
  if ~isfield(p,'tempcoeff'),  p.tempcoeff = '+';             end

  % Expand upper/lower bounds for activation energy.
  if startsWith(p.tempfcn,'Eact')
    parts = strsplit(p.tempfcn,':');
    nargs = length(parts)-1;
    if nargs==2
      % UB and LB given - use default INIT.
      if ~isfield(p,'EactLB'); p.EactLB = str2double(parts{2})*1000; end % convert kJ to J
      if ~isfield(p,'EactUB'); p.EactUB = str2double(parts{3})*1000; end
      if ~isfield(p,'EactINIT'); p.EactINIT = 0; end
    elseif nargs==3
      % UB, LB, and INIT given.
      if ~isfield(p,'EactLB'); p.EactLB = str2double(parts{2})*1000; end % convert kJ to J
      if ~isfield(p,'EactUB'); p.EactUB = str2double(parts{3})*1000; end
      if ~isfield(p,'EactINIT'); p.EactINIT = str2double(parts{4})*1000; end
    else
      % No arguments - use defualts.
      if ~isfield(p,'EactLB'); p.EactLB = 0; end % J
      if ~isfield(p,'EactUB'); p.EactUB = 10000; end  % J
      if ~isfield(p,'EactINIT'); p.EactINIT = 0; end  % J
    end % if
    % Remove additional arguments.
    p.tempfcn = 'Eact';
  end % if

  % Indicates this structure should not be flattened by FLATTENSTRUCT()
  p.noflatten__ = true;
end

