% PACK Stuff a model structure into a vector.
%
% vect = PACK(model,metadata) converts the model structure MODEL into
%   vector VECT for use with optimization routines. META is generated
%   using MODELSPEC().
%
% vect = PACK(...,defaultval) sets the default value of missing
%   parameters instead of throwing an error.
%
% The following automatically convert the lb, ub, and init fields from
% the param() structures if available:
%
% vect = PACK('lb',metadata)
% vect = PACK('ub',metadata)
% vect = PACK('init',metadata)
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

function [vect, metadata] = pack(model,metadata,default,varargin)

  if ~exist('default','var')
    default = [];
  end
  coerce = any(strcmp(varargin,'coerce'));
  notemp = any(strcmp(varargin,'notemp'));
  nolog = any(strcmp(varargin,'nolog'));

  paramnames = fieldnames(metadata.params);
  if ischar(model)
    % Assemble model by fetching the value of the specified field.
    fieldname = model;
    if ~any(strcmp(fieldname,{'lb','ub','init'}))
        error('Specify lb, ub, or init as first char string argument.');
    end
    model = struct;
    for k = 1:length(paramnames)
      paramname = paramnames{k};
      meta = metadata.params.(paramname);
      if isfield(meta,fieldname) % may not have field if fixed param
        model.(paramname) = meta.(fieldname);
        % Add additional activation energy parameter if needed.
        if strcmpi(meta.tempfcn,'Eact')
          fieldnameEact = ['Eact' upper(fieldname)];
          paramnameEact = [paramname '_Eact'];
          model.(paramnameEact) = metadata.params.(paramname).(fieldnameEact);
        end % if
      end
    end % for
  else
    % Flatten model.
    model = optutil.flattenstruct(model);
  end

  % Construct parameter vector.
  vect = zeros(metadata.nvars,1);
  
  cursor = 1;
  for k = 1:length(paramnames)
    paramname = paramnames{k};
    meta = metadata.params.(paramname);

    if isfield(meta,'fix') && ~any(meta.fixmask)
      % Fixed parameter, skip packing.
      continue;
    end

    % Regular parameter.
    if isfield(model,paramname)
      value = model.(paramname);
    elseif ~isempty(default)
      value = repmat(default,meta.len,1);
    else
      error('Could not resolve value for parameter %s',paramname);
    end
    if isfield(meta,'logscale') && meta.logscale == true && ~nolog
      % Encode logarithmic value.
      value = log10(value);
    end
    [vallen, valmult] = size(value);

    % Determine required multiplicity of the parameter (>1 when 
    % temperature dependence is modeled using LUT approach).
    mult = 1;
    if strcmpi(meta.tempfcn,'lut')
     mult = metadata.ntemps;
    end

    % Expand scalar parameters to correct size.
    if coerce && vallen == 1 && valmult == 1
      value = value*ones(meta.len,mult);
    end

    % Remove fixed components of the value (if it is a vector).
    if isfield(meta,'fix')
      value = value(meta.fixmask);
      len = sum(meta.fixmask);
    else
      len = meta.len;
    end

    % Place value into parameter vector.
    [vallen, valmult] = size(value);
    if vallen == len && valmult == mult
      vect(cursor:cursor+len*mult-1,1) = value(:);
    else
      if vallen ~= len
        error(['Parameter %s has incorrect length ' ...
            '(metadata says it should be %d, but it is actualy %d)'], ...
            paramname,len,vallen);
      else
        error(['Parameter %s has incorrect multiplicity ' ...
            '(metadata says it should be %d, but it is actualy %d)'], ...
            paramname,mult,valmult);
      end
    end
    cursor = cursor + len*mult;

    % Check for activation energy parameter.
    if ~notemp && strcmpi(meta.tempfcn,'Eact')
      % Should exist - add Eact to parameter vector.
      paramnameEact = [paramname '_Eact'];
      if ~isfield(model,paramnameEact)
        error(['Eact not found for parameter %s. ' ...
            'Hint: Specify an activation energy parameter %s.'], ...
            paramname,paramnameEact);
      end
      Eact = model.(paramnameEact);
      [Eactlen, Eactmult] = size(Eact);
      if Eactmult ~= 1
        error('%s must have multiplicity 1.',paramnameEact);
      end
      % Expand scalar Eact to correct size.
      if coerce && Eactlen == 1
        Eact = Eact.*ones(len,1);
        Eactlen = len;
      end
      if Eactlen ~= len
        error('%s must have length %d.',paramnameEact,len);
      end
      vect(cursor:cursor+len-1,1) = Eact(:);
      cursor = cursor + len;
    end
  end % for
end