% FLATTENSTRUCT Recursively flatten a structure with hierarchy.
%
% flat = FLATTENSTRUCT(s) converts the hierarchical structure S
%   into flat structure FLAT by joining nested field names with
%   the double underscore (__) delimiter.
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         
function flat = flattenstruct(s, prefix, whitelist)

  flat = struct;

  if ~exist('prefix','var')
      prefix = '';
  end
  if ~exist('whitelist','var')
      whitelist = {};
      hasWhitelist = false;
  else
      hasWhitelist = true;
  end
  if ~isstruct(s)
    flat.(prefix) = s;
    return;
  end

  fnames = fieldnames(s);
  if any(strcmp('noflatten__',fnames))
    flat.(prefix) = s;
    return;
  end

  for k = 1:length(fnames)
    fname = fnames{k};
    if hasWhitelist && ~any(strcmp(fname,whitelist))
      % Skip this field.
      continue;
    end
    value = s.(fname);
    if strcmp(prefix,'')
      flatName = fname;
    else
      flatName = [prefix '__' fname];
    end
    sflat = optutil.flattenstruct(value,flatName);
    subfnames = fieldnames(sflat);
    for j = 1:length(subfnames)
      subfname = subfnames{j};
      flat.(subfname) = sflat.(subfname);
    end
  end % for
end