% example5.m
%
% Demonstrates how to run the EIS optimizations.
%
% By default, this code starts a graphical user interface (GUI) that allows
% the user to experiment with various parameters to see how changing their
% values affects the Nyquist plots of impedance (or admittance). 
% Adjust the initial guesses at will; start the optimization; when done,
% simply close the GUI.
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

clear; close all; clc;
if(~isdeployed),cd(fileparts(which(mfilename))); end
addpath(genpath('..'));
rng(42); % make results repeatable

% Set output level of toolbox functions.
%outp.quiet;   % warnings and errors only
%outp.normal;  % text output only
outp.verbose; % text and plot output

% Configure debug output of toolbox functions.
outp.debug('off');  % 'on' or 'off'

% Choose cell dataset.
cellspec = labcell('PAN');


% Read raw data into MAT files --------------------------------------------
makeMATfileEIS(cellspec);


% Process raw data from MAT files -----------------------------------------
outp.normal;
configEIS.preprocessfn = @preprocessImpedance; % preprocess fn (dfn below)
processData = processEIS(cellspec,configEIS);


% Regress model to lab data -----------------------------------------------
outp.verbose;
import('optutil.param');
JNEG = length(processData.cellData.neg.U0); % number of MSMR galleries (neg)
JPOS = length(processData.cellData.pos.U0); % number of MSMR galleries (pos)
R = 8.3144598;   % Molar gas constant [J/mol K]
F = 96485.3329;  % Faraday constant [C/mol]
tplus0 = 0.4;    % Li+ transference number (guess for estimating psi)
% Build model struct defining EIS regression
p = processData.cellData;   % struct of initial param estimates
m = struct;                 % model struct (built below)
QAh = p.const.Q;            % cell capacity in Ah
% Fixed parameters (weakly- or non-identifiable)
m.const.psi = param('fix',R/F/(1-tplus0));   % Nernst-Einstein value
m.neg.alpha = param('fix',0.5*ones(JNEG,1));
m.pos.alpha = param('fix',0.5*ones(JPOS,1));
% Free parameters
m.const.kD  = param('init',-8.62e-5,    'lb',-1.55e-2,  'ub',-10.62e-5);
m.const.Rc  = param('init',1e-5,        'lb',1e-9,      'ub',0.1,   'logscale',true);
m.neg.sigma = param('init',p.neg.sigma, 'lb',1e5,       'ub',1e8,   'logscale',true);
m.neg.kappa = param('init',p.neg.kappa, 'lb',88,        'ub',8000,  'tempfcn','Eact:0:100','EactINIT',p.neg.kappa_Eact);
m.neg.Rdl   = param('init',p.neg.Rdl,   'lb',1e-8,      'ub',1e-1,  'tempfcn','Eact:0:100','EactINIT',p.neg.Rdl_Eact,'tempcoeff','-','logscale',true);
m.neg.Rf    = param('init',p.neg.Rf,    'lb',1e-6,      'ub',1e-2,  'tempfcn','Eact:0:100','EactINIT',p.neg.Rf_Eact,'tempcoeff','-','logscale',true);
m.neg.k0    = param('init',p.neg.k0,    'lb',1e-10,     'ub',1e10,  'tempfcn','Eact:0:100','EactINIT',p.neg.k0_Eact,'logscale',true,'len',JNEG);
m.neg.qe    = param('init',QAh/100,     'lb',QAh/400,   'ub',3*QAh/14,'logscale',true);
m.neg.Cdl   = param('init',0.01,        'lb',0.001,     'ub',100,   'logscale',true);
m.neg.nDL   = param('init',1,           'lb',0.9,       'ub',1);
m.neg.wDL   = param('init',1e-5,        'lb',1e-6,      'ub',1e-3,  'logscale',true);
m.neg.nF    = param('init',1,           'lb',0.9,       'ub',1);
m.neg.Dsref = param('init',p.neg.Dsref, 'lb',1e-5,      'ub',0.1,   'tempfcn','Eact:0:100','EactINIT',0,'logscale',true);
m.sep.kappa = param('init',p.sep.kappa, 'lb',362,       'ub',10000, 'tempfcn','Eact:0:100','EactINIT',p.sep.kappa_Eact);
m.sep.qe    = param('init',QAh/100,     'lb',QAh/800,   'ub',QAh/50,'logscale',true);
m.pos.sigma = param('init',p.pos.sigma, 'lb',1e4,       'ub',1e8);
m.pos.kappa = param('init',p.pos.kappa, 'lb',68,        'ub',1369,  'tempfcn','Eact:0:100','EactINIT',p.pos.kappa_Eact);
m.pos.Rdl   = param('init',p.pos.Rdl,   'lb',1e-6,      'ub',1e-1,  'tempfcn','Eact:0:100','EactINIT',p.pos.Rdl_Eact,'tempcoeff','-','logscale',true);
m.pos.Rf    = param('init',p.pos.Rf,    'lb',1e-10,     'ub',1e-2,  'tempfcn','Eact:0:100','EactINIT',p.pos.Rf_Eact,'tempcoeff','-','logscale',true);
m.pos.k0    = param('init',p.pos.k0,    'lb',1e-10,     'ub',1e10,  'tempfcn','Eact:0:100','EactINIT',p.pos.k0_Eact,'logscale',true,'len',JPOS);
m.pos.qe    = param('init',QAh/100,     'lb',QAh/400,   'ub',3*QAh/14,'logscale',true);
m.pos.Cdl   = param('init',0.01,        'lb',0.001,     'ub',100,   'logscale',true);
m.pos.nDL   = param('init',1,           'lb',0.9,       'ub',1);
m.pos.wDL   = param('init',1e-5,        'lb',1e-6,      'ub',1e-2);
m.pos.nF    = param('init',1,           'lb',0.9,       'ub',1);
m.pos.Dsref = param('init',p.pos.Dsref, 'lb',1e-5,      'ub',0.1,  'tempfcn','Eact:0:100','EactINIT',0,'logscale',true);
% Run TF model regression.
fitEIS(cellspec,m,'25degC','gui','Np',5000,'filterFcn',@filterEIS);



% UTILITY FUNCTIONS -------------------------------------------------------

function eis = filterEIS(eis)
%FILTEREIS Filters EIS data at each temperature setpoint.

% Ignore data at or below 5% SOC.
indz = eis.socPct > 5;
eis.freq = eis.freq(:,indz);
eis.Z = eis.Z(:,indz);
eis.socPct = eis.socPct(indz);
% !! Important: Apply SOC filtering to OCP fields.
fields = {'thetaNEG','UocpNEG','dUocpNEG','thetaPOS','UocpPOS','dUocpPOS'};
for kf = 1:length(fields)
    field = fields{kf};
    eis.(field) = eis.(field)(indz);
end

% Ignore data below 1Hz.
% indf = eis.freq(:,1)>=0.1;
% eis.freq = eis.freq(indf,:);
% eis.Z = eis.Z(indf,:);
end

function [Znew,freqNew] = preprocessImpedance(Z,freq,~,~)
%PREPROCESSIMPEDANCE Pre-process impedance measurements.
% Currently fits an ECM for inductance to high-frequency measurements and 
% extrapolates the inductance model predictions over higher frequency.

% Ignore data below 1mHz.
ind = freq>=1e-3;
freq = freq(ind);
Z = Z(ind);

s = 1j*2*pi*freq;
        
% Setup initial values and boundaries.
x0  = [min(real(Z));1e-7;1e-7;1e-7;1e-5;1e-5]; % Rinf,Linf,L1,L2,tau1,tau2
lb  = [min(real(Z))*0.5;ones(5,1)*1e-10];
ub  = [min(real(Z))*1.0;ones(5,1)*1e-2];

% Only use 100kHz-10kHz measurements in fitting the ECM.
ind = 10e3<=freq&freq<=100e3;

% Setup the ECM cost function.
Zmodel = @(x) x(1) + s(ind)*x(2) + ...
     s(ind)*x(3)./(1+s(ind)*x(5)) + ...
     s(ind)*x(4)./(1+s(ind)*x(6));
Zmeas = Z(ind);
Jre = @(x) (real(Zmodel(x))./real(Zmeas)-1).^2;
Jim = @(x) (imag(Zmodel(x))./imag(Zmeas)-1).^2;
J   = @(x) sum(Jre(x))*10+sum(Jim(x));

% Run optimization.
opt = optimset('Display','off',...
    'TolFun',1e-15,'TolX',1e-15,'MaxFunEvals',1e6,'MaxIter',1e6);
xEst = fminsearchbnd(J,x0,lb,ub,opt);

% Extrapolate inductance model over high-frequency.
ff = 10.^(linspace(log10(max(freq))+1,log10(max(freq))*1.01,20)); 
ff = ff(:);
ss = 1j*2*pi*ff;
ZLext = xEst(1) + ss*xEst(2) ...
    + ss*xEst(3)./(1+ss*xEst(5)) ...
    + ss*xEst(4)./(1+ss*xEst(6));

% Output result.
% Ensure frequency vector is sorted.
freqNew = [ff(:);freq(:)];
Znew = [ZLext(:);Z(:)];
[freqNew,indSort] = sort(freqNew);
Znew = Znew(indSort);

end