% example4.m
%
% Demonstrates how to run the pulse-test optimizations.
% 
% - The regression takes place over two steps. The first uses
%   linear methods and is slow. The second seeks to enhance the
%   results from the first optimization and uses nonlinear
%   methods and is slower (hours). Patience is advised!
% - During regression, the RMSE value that is output is updated
%   whenever a better set of parameters is found and gives
%   some indication of optimization progress.
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

clear; close all; clc;
if(~isdeployed),cd(fileparts(which(mfilename))); end
addpath(genpath('..'));
rng(42); % make results repeatable

% Set output level of toolbox functions.
%outp.quiet;   % warnings and errors only
%outp.normal;   % text output only
outp.verbose; % text and plot output

% Configure debug output of toolbox functions.
outp.debug('off');  % 'on' or 'off'

% Choose cell dataset.
cellspec = labcell('PAN');


% Read raw data into MAT files --------------------------------------------
matDat = makeMATfilePulse(cellspec);


% Process raw data from MAT files -----------------------------------------
configPULSE.vtpct = 25;  % edge threshold voltage, pct peak under/overshoot
configPULSE.ss = [50e-6 195e-6]; % steady-state interval [sec]
configPULSE.relax = [260e-6 350e-6]; % relaxation interval [sec]
configPULSE.regress = [-10e-6 250e-6]; % ECM regression interval [sec]
configPULSE.getWeight = @getWeight; % residual weighting function
configPULSE.postProcessECM = @postProcessECM; % post-processing function
processData = processPulse(cellspec,configPULSE);


% Regress model to lab data -----------------------------------------------
% Build model. Type `help optutil.param` for more information.
import('optutil.param');
JNEG = length(processData.cellData.neg.U0); % number of MSMR galleries (neg)
JPOS = length(processData.cellData.pos.U0); % number of MSMR galleries (pos)
m = struct;
m.neg.sigma = param('init',3.77e5,  'lb',1e5,   'ub',5e5);
m.neg.kappa = param('init',873,     'lb',273,   'ub',5457,  'tempfcn','Eact:0:100');
m.neg.Rdl   = param('init',1e-5,    'lb',1e-6,  'ub',1e-1,  'tempfcn','Eact:0:100','tempcoeff','-');
m.neg.Rf    = param('init',1e-6,    'lb',1e-6,  'ub',1e-2,  'tempfcn','Eact:0:100','tempcoeff','-');
m.neg.k0    = param('init',1e-3,    'lb',1e-10, 'ub',1e10,  'tempfcn','Eact:0:100','logscale',true,'len',JNEG);
m.sep.kappa = param('init',1159,    'lb',362,   'ub',7245,  'tempfcn','Eact:0:100');
m.pos.sigma = param('init',1.52e4,  'lb',1e4,   'ub',1.58e4);
m.pos.kappa = param('init',219,     'lb',68,    'ub',1369,  'tempfcn','Eact:0:100');
m.pos.Rdl   = param('init',1e-5,    'lb',1e-6,  'ub',1e-1,  'tempfcn','Eact:0:100','tempcoeff','-');
m.pos.Rf    = param('init',1e-6,    'lb',1e-9,  'ub',1e-2,  'tempfcn','Eact:0:100','tempcoeff','-');
m.pos.k0    = param('init',1e-3,    'lb',1e-10, 'ub',1e10,  'tempfcn','Eact:0:100','logscale',true,'len',JPOS);
m.neg.alpha = param('fix',0.5*ones(JNEG,1));
m.pos.alpha = param('fix',0.5*ones(JPOS,1));
fitPulse(cellspec,m,'5degC 15degC 25degC','Np',10000);


% UTILITY FUNCTIONS -------------------------------------------------------

%GETWEIGHT Get weight vector for ECM regression given the time vector.
function w = getWeight(time)
  w = 5*ones(size(time));
  w(time>60e-6)  = 50;     % put more attention on steady-state
  w(time>190e-6) = 1;
end

%POSTPROCESSECM Post-process ECM parameter values.
function ecmdat = postProcessECM(ecmdat)
  % Fudge factors to remove offset observed when changing scales.
  I  = ecmdat.iapp;
  R0 = ecmdat.R;    % dim1=SOC, dim2=I
  % Modify R0 by subtracting Rx.
  I3p = I>60;         I3n = I<-60;
  I2p = I>30 & I<=60; I2n = I>=-60 & I<-30;
  I1p = I>5  & I<=30; I1n = I>=-30 & I<-5 ;
  R3p = max(R0(:,I3p),[],2); 
  R2p = (median(R0(:,I2p),2)+min(R0(:,I2p),[],2))/2; Rxp32 = R3p-R2p;
  R3n = max(R0(:,I3n),[],2); R2n = (median(R0(:,I2p),2)+min(R0(:,I2p),[],2))/2; Rxn32 = R3n-R2n;
  R2p = max(R0(:,I2p),[],2); R1p = (median(R0(:,I1p),2)+min(R0(:,I1p),[],2))/2; Rxp21 = R2p-R1p;
  R2n = max(R0(:,I2n),[],2); R1n = (median(R0(:,I1n),2)+min(R0(:,I1n),[],2))/2; Rxn21 = R2n-R1n;
  Rx = mean([Rxp32 Rxn32 Rxp21 Rxn21],2);
  R0(:,I2p) = R0(:,I2p) - 0.75*Rx;
  R0(:,I2n) = R0(:,I2n) - 0.75*Rx;
  R0(:,I3p) = R0(:,I3p) - 1.5*Rx;
  R0(:,I3n) = R0(:,I3n) - 1.5*Rx;
  % Save result.
  ecmdat.R = R0;
end