% This example demonstrates how to run the OCP optimizations. 
%
% It first calls "makeMATfileOCP" to convert the raw .xlsx data files
% collected  using our Arbin equipment into .mat files for easier access.
% If the .mat file already exists (indicated by a "lock" file in the
% destination folder), then this step is skipped unless forced. 
%
% Next, it calls "processOCP" to perform some initial processing of the
% data to calibrate it for the next step.
%
% Finally, it calls "fitOCP" to fit an MSMR model to the OCP data.
%
% Copyright (©) 2024 The Regents of the University of Colorado, a body
% corporate. Created by Gregory L. Plett and M. Scott Trimboli of the
% University of Colorado Colorado Springs (UCCS). This work is licensed
% under a Creative Commons "Attribution-ShareAlike 4.0 International" Intl.
% License. https://creativecommons.org/licenses/by-sa/4.0/ 
% This code is provided as a supplement to: Gregory L. Plett and M. Scott
% Trimboli, "Battery Management Systems, Volume III, Physics-Based
% Methods," Artech House, 2024. It is provided "as is", without express or
% implied warranty. Attribution should be given by citing: Gregory L. Plett
% and M. Scott Trimboli, Battery Management Systems, Volume III:
% Physics-Based Methods, Artech House, 2024.         

clear; close all; clc;
if(~isdeployed),cd(fileparts(which(mfilename))); end
addpath(genpath('../../TOOLBOX'));

% Set output level of toolbox functions.
%outp.quiet;  % warnings and errors only
%outp.normal; % text output only
outp.verbose; % text and plot output

% Configure debug output of toolbox functions.
outp.debug('off');  % 'on' or 'off'

% Choose cell dataset.
cellspec = labcell('PAN'); 

% Read raw data into MAT files --------------------------------------------
makeMATfileOCP(cellspec);

% Process raw data from MAT files -----------------------------------------
processOCP(cellspec);

% Regress processed data to MSMR model ------------------------------------

% Negative electrode configuration. Initial MSMR values from: Mark
% Verbrugge et al. 2017 J. Electrochem. Soc. 164 E3243 
configNEG.vmin = 0.01;
configNEG.vmax = 1;
configNEG.init.U0 = [0.08843;0.12799;0.14331;0.16984;0.21446;0.36325];
configNEG.init.X = [0.43336;0.23963;0.15018;0.05462;0.06744;0.0548];
configNEG.init.omega = [0.08611;0.08009;0.72469;2.53277;0.09470;5.97354];
configNEG.init.thetamin = 0.01;
configNEG.init.thetamax = 0.97;
configNEG.lb.U0 = configNEG.init.U0*0.9;
configNEG.lb.X = configNEG.init.X*0.95;
configNEG.lb.omega = configNEG.init.omega*0.9;
configNEG.lb.thetamin = 1e-5;
configNEG.lb.thetamax = 0.95;
configNEG.ub.U0 = configNEG.init.U0/0.9;
configNEG.ub.X = configNEG.init.X/0.95;
configNEG.ub.omega = configNEG.init.omega/0.9;
configNEG.ub.thetamin = 0.03;
configNEG.ub.thetamax = 1-1e-5;
configNEG.optw = 1e-3;
configNEG.datype = 'useAvg';
configNEG.daxcorrfiltz = @(Z){
    0.0<Z & Z<0.3  % focus on the first "stair"
    0.8<Z & Z<1.0  % focus on the last "stair"
};

% Positive electrode configuration. Initial MSMR values from: Mark
% Verbrugge et al. 2017 J. Electrochem. Soc. 164 E3243 
configPOS.vmin = 3;
configPOS.vmax = 4.3;
configPOS.init.U0 = [3.64;3.72645;3.90575;4.22955];
configPOS.init.X = [0.13442;0.32460;0.21118;0.3298];
configPOS.init.omega = [0.96710;1.39712;3.50500;5.52757];
configPOS.init.thetamin = 0.1274;
configPOS.init.thetamax = 0.98;
configPOS.lb.U0 = configPOS.init.U0*0.97;
configPOS.lb.X = configPOS.init.X*0.95;
configPOS.lb.omega = configPOS.init.omega*0.95;
configPOS.lb.thetamin = 0.1;
configPOS.lb.thetamax = 0.97;
configPOS.ub.U0 = configPOS.init.U0/0.97;
configPOS.ub.X = configPOS.init.X/0.95;
configPOS.ub.omega = configPOS.init.omega/0.95;
configPOS.ub.thetamin = 0.2;
configPOS.ub.thetamax = 1-1e-8;
configPOS.optw = 1e-3;
configPOS.datype = 'useAvg';
configPOS.daxcorrfiltz = @(Z){
    0.05<Z & Z<0.9
    0.00<Z & Z<0.9
};

% Perform model regression.
fitOCP(cellspec,configNEG,configPOS);
